<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AppHealthBundle;

use Cyber\AppHealthBundle\Model\SimpleHealthStatus;
use Exception;

class HealthStatusFetcher
{
    /**
     * @var HealthStatusReducer
     */
    private $reducer;

    /**
     * HealthStatusFetcher constructor.
     *
     * @param HealthStatusReducer $reducer
     */
    public function __construct(HealthStatusReducer $reducer)
    {
        $this->reducer = $reducer;
    }

    /**
     * Fetches all status from the url and converts them into an array of HealthStatusInterface objects.
     *
     * @param string $url the URL from which to fetch health data
     *
     * @return HealthStatusInterface[]
     */
    public function fetchAll(string $url): array
    {
        try {
            // silence errors to prevent symfony throwing internal exceptions
            $content = @\file_get_contents($url);
            if (false === $content) {
                throw new Exception('No data returned in response from HealthCheck request.');
            }

            $data     = \json_decode($content, true);
            $statuses = [];

            foreach ($data as $healthReport) {
                $statuses[] = new SimpleHealthStatus($healthReport['resource'], $healthReport['status'], $healthReport['message'] ?? null, $healthReport['extra'] ?? null);
            }

            if (empty($statuses)) {
                throw new Exception('HealthCheck response had no data about any resources');
            }

            return $statuses;
        } catch (\Exception $e) {
            return [new SimpleHealthStatus('all', HealthStatusInterface::STATUS_UNKNOWN, $e->getMessage())];
        }
    }

    /**
     * Fetches all status from the url and converts them into an array of HealthStatusInterface objects.
     *
     * Then compresses all objects into one. The status of the returned object will be the most severe status found
     * within the list. And the message will be concatenation of all messages from all statuses in the list.
     *
     * @param string $reducedResource the new resource name to assign to the reduced resource
     * @param string $url             the URL from which to fetch health data
     *
     * @return HealthStatusInterface
     */
    public function fetchReduced(string $reducedResource, string $url): HealthStatusInterface
    {
        $statuses = $this->fetchAll($url);

        return $this->reducer->reduce($reducedResource, $statuses);
    }
}
