<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AppHealthBundle;

use Cyber\AppHealthBundle\HealthStatusFetcher;
use Cyber\AppHealthBundle\HealthStatusInterface;
use Cyber\AppHealthBundle\HealthStatusReducer;
use Cyber\AppHealthBundle\Model\SimpleHealthStatus;
use PHPUnit\Framework\TestCase;

/**
 * @covers \Cyber\AppHealthBundle\HealthStatusFetcher
 */
class HealthStatusFetcherTest extends TestCase
{
    /**
     * @var HealthStatusReducer|\PHPUnit\Framework\MockObject\MockObject
     */
    private $reducer;

    /**
     * @var HealthStatusFetcher
     */
    private $instance;

    protected function setUp(): void
    {
        $this->reducer = $this->getMockBuilder(HealthStatusReducer::class)->getMock();

        $this->instance = new HealthStatusFetcher($this->reducer);
    }

    public function testFetch(): void
    {
        $reduced = new SimpleHealthStatus('test-res', 1);
        $this->reducer->expects($this->once())
            ->method('reduce')
            ->with('test', $this->callback(function ($value) {
                $this->assertCount(2, $value);
                $this->assertContainsOnlyInstancesOf(SimpleHealthStatus::class, $value);

                $expected = [
                    'database',
                    'amazon_s3',
                ];
                foreach ($value as $i => $status) {
                    // @var SimpleHealthStatus $status
                    $this->assertEquals($expected[$i], $status->getResource());
                    $this->assertEquals($i, $status->getStatus());
                    $this->assertEquals('msg' . $i, $status->getMessage());
                    $this->assertEquals(['extra' . $i], $status->getExtra());
                }

                return true;
            }))
            ->willReturn($reduced);
        $result = $this->instance->fetchReduced('test', __DIR__ . '/health_response_healthy.json');

        $this->assertSame($reduced, $result);
    }

    public function testNoResponse(): void
    {
        $result = $this->instance->fetchAll(__DIR__ . '/missing_file.json');
        $this->assertContainsOnlyInstancesOf(SimpleHealthStatus::class, $result);
        $this->assertCount(1, $result);
        $this->assertEquals('all', $result[0]->getResource());
        $this->assertEquals(HealthStatusInterface::STATUS_UNKNOWN, $result[0]->getStatus());
        $this->assertEquals('No data returned in response from HealthCheck request.', $result[0]->getMessage());
    }

    public function testEmptyResponse(): void
    {
        $result = $this->instance->fetchAll(__DIR__ . '/health_response_empty.json');
        $this->assertContainsOnlyInstancesOf(SimpleHealthStatus::class, $result);
        $this->assertCount(1, $result);
        $this->assertEquals('all', $result[0]->getResource());
        $this->assertEquals(HealthStatusInterface::STATUS_UNKNOWN, $result[0]->getStatus());
        $this->assertEquals('HealthCheck response had no data about any resources', $result[0]->getMessage());
    }
}
