<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AppHealthBundle;

use Cyber\AppHealthBundle\Model\SimpleHealthStatus;
use Throwable;

class HealthStatusCollector
{
    /**
     * @var CheckerRegistryService
     */
    private $checkerRegistry;

    /**
     * @var HealthStatusFetcher
     */
    private $fetcher;

    /**
     * @var string[]
     */
    private $urls = [];

    /**
     * HealthStatusCollector constructor.
     *
     * @param CheckerRegistryService $checkerRegistry
     */
    public function __construct(CheckerRegistryService $checkerRegistry, HealthStatusFetcher $fetcher)
    {
        $this->checkerRegistry = $checkerRegistry;
        $this->fetcher         = $fetcher;
    }

    public function addUrl(string $resource, string $url): void
    {
        $this->urls[$resource] = $url;
    }

    /**
     * @return HealthStatusInterface[]
     */
    public function collect(): array
    {
        $report = [];

        $checkers = $this->checkerRegistry->getHealthCheckers();
        foreach ($checkers as $checker) {
            $resource = $this->extractResource($checker);

            try {
                $report[] = $checker->getHealthStatus();
            } catch (Throwable $ex) {
                $report[] = new SimpleHealthStatus(
                    $resource,
                    HealthStatusInterface::STATUS_DOWN,
                    'Failed to retrieve health data.',
                    [$ex->getMessage()]
                );
            }
        }

        foreach ($this->urls as $resource => $url) {
            $report[] = $this->fetcher->fetchReduced($resource, $url);
        }

        return $report;
    }

    private function extractResource(HealthCheckerInterface $checker): string
    {
        if (\method_exists($checker, 'getResourceName')) {
            return $checker->getResourceName();
        }

        $class = \get_class($checker);
        trigger_deprecation(
            'cyber/app-health-bundle',
            '5.2',
            'You must define getResourceName() function in %s',
            $class
        );

        return $class;
    }
}
