# Configuration

If your application depends on services which also utilize this bundle for health reporting. 
You may provide urls for those services in your config, so that your health report includes
the summary of health reports of dependent services.

```yaml
cyber_app_health:
    external_urls:
        vantage: 'https://vantage.host.name/health'
        warehouse: 'https://warehouse.host.name/health'
```

# Routing

Expose the bundle health controller in your routes, so that you could query the application 
health.

> The controller reports its responses as cacheable, so you should have a proper caching reverse
> proxy configured to ease the load off your server, if you are going to query application health
> frequently. You can use [CacheKernel](https://symfony.com/doc/current/http_cache.html) as a starting point

```yaml
# routing.yml
app_health_route:
    path: /health
    controller: CyberAppHealthBundle:SimpleHealth:getHealth
```

# Health Checkers

Create your health checkers which will be responsible for checking individual parts of your application.
Report from each checker will be returned as a separate object in the response from the health controller.

To create a health checker you have to implement `Cyber\AppHealthBundle\HealthCheckerInterface` and register it as 
as service. 

> In symfony versions which support `autoconfigure`, use it to configure the checker automatically.

> In symfony versions that do not support `autoconfigure`, tag your service with `cyber.health.checker`.

Here is an example health checker which reports the database status:

```php
<?php
namespace App\Health;

use Cyber\AppHealthBundle\HealthCheckerInterface;
use Cyber\AppHealthBundle\HealthStatusInterface;
use Cyber\AppHealthBundle\SimpleHealthStatus;

class DbHealthChecker implements HealthCheckerInterface
{

    /** @var \Doctrine\DBAL\Connection  */
    private $connection;
    
    /**
     * @param \Doctrine\DBAL\Connection $connection inject the database connection
     */
    public function __construct($connection) {
        $this->connection = $connection;
    }

    public function getHealthStatus(): HealthStatusInterface
    {
        if($this->connection->ping()){
            $status = new SimpleHealthStatus('database', HealthStatusInterface::STATUS_HEALTHY);            
        } else {
            $status = new SimpleHealthStatus('database', HealthStatusInterface::STATUS_DOWN, 'Database is down');
        }
        
        return $status;
    }
}
```

> You can use the bundle supplied `SimpleHealthStatus` class, or if you need something more elaborate
> define your own class which implements `HealthStatusInterface`

# Manual fetching of remote statuses

If you are coding a system which will fetch statues from other systems and do something with them. You may
want to look into `Cyber\AppHealthBundle\HealthStatusFetcher` service. This services can fetch health status response
generated by health controller and parse them into usable objects. It can also reduce an array of
statuses into a single summary status. 
