<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AppHealthBundle;

interface HealthStatusInterface
{
    /**
     * Resource is functioning normally. Message is typically null.
     */
    public const STATUS_HEALTHY = 0;

    /**
     * Status could not be determined. Message should contain explanation why status could not be determined.
     */
    public const STATUS_UNKNOWN = 100;

    /**
     * Resource is functioning normally, but has an informational message attached. Message should contain the info text.
     */
    public const STATUS_INFO = 200;

    /**
     * Resource is experiencing some issues, but is still more or less functional. Message should contain warning text.
     */
    public const STATUS_WARNING = 300;

    /**
     * Resource is experiencing critical failures and is not operational. Message should contain any extra info if applicable.
     */
    public const STATUS_DOWN = 400;

    public const LABELS = [
        self::STATUS_HEALTHY => 'Healthy',
        self::STATUS_INFO    => 'Info',
        self::STATUS_WARNING => 'Warning',
        self::STATUS_DOWN    => 'Down',
        self::STATUS_UNKNOWN => 'Unknown',
    ];

    /**
     * @return string the resource name this health result pertains to
     */
    public function getResource(): string;

    /**
     * Returns the health status of the resource. Returned value should be one of the HealthStatusInterface::STATUS_*
     * constants.
     *
     * @return int the health status of the resource
     */
    public function getStatus(): int;

    /**
     * A message clarifying the status. See the STATUS_* constants for recommendation on when the message should be
     * provided.
     *
     * @return null|string
     */
    public function getMessage(): ?string;

    /**
     * This array will be included directly into the health report. You can you this to customize what data is in
     * the report for personal use. This may or may not be ignored by our central health monitoring software.
     *
     * @return null|array<mixed> extra data to include into health report
     */
    public function getExtra(): ?array;
}
