<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AppHealthBundle;

use Cyber\AppHealthBundle\Model\SimpleHealthStatus;
use InvalidArgumentException;

class HealthStatusReducer
{
    /**
     * Reduces a list of HealthStatusInterface to a single instance. The returned instance will have the most critical
     * status value and a message containing concatenation of all messages from all HealthStatusInterface.
     *
     * This is useful if your app depends on another app which uses this same exact health report. If the dependent
     * app returns info about multiple resources, this function can be used to reduce that to a single status
     * to use as a summary in your health report, instead of reporting each of the services used by the dependent app.
     *
     * @param string                  $resource resource name to assigned to returned HealthStatusInterface
     * @param HealthStatusInterface[] $statuses a list of statuses to reduce
     *
     * @return HealthStatusInterface
     */
    public function reduce(string $resource, array $statuses): HealthStatusInterface
    {
        $finalStatus = HealthStatusInterface::STATUS_HEALTHY;
        $messages    = [];

        foreach ($statuses as $status) {
            if (!($status instanceof HealthStatusInterface)) {
                throw new InvalidArgumentException('All statuses must be instance of HealthStatusInterface');
            }
            // the bigger the value the more critical the status is, we want to get most critical status
            $finalStatus = \max($finalStatus, $status->getStatus());
            $message     = $status->getMessage();

            if ($message) {
                $messages[] = \sprintf('%s => %s', $status->getResource(), $message);
            }
        }

        return new SimpleHealthStatus($resource, $finalStatus, \implode('; ', $messages));
    }
}
