<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Cyber\AuditBundle\Describer\Describer;
use Cyber\AuditBundle\Describer\ToStringDescriber;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Persistence\Mapping\ClassMetadata;
use InvalidArgumentException;
use LogicException;
use ReflectionProperty;

class AuditMetadata
{
    /** @var ArrayCollection<string, ReflectionProperty>|ReflectionProperty[] */
    private $owners;

    /** @var ArrayCollection<int, string>|string[] */
    private $properties;

    /** @var array<array<string>> */
    private $userValues = [];

    /** @var ClassMetadata */
    private $classMetadata;

    /** @var bool */
    private $trackSoftDelete = false;

    /** @var string */
    private $alias;

    /** @var Describer */
    private $describer;

    /** @var string[] */
    private $embeddedFields;

    public function __construct(ClassMetadata $classMetadata)
    {
        $this->owners        = new ArrayCollection();
        $this->properties    = new ArrayCollection();
        $this->classMetadata = $classMetadata;
    }

    public function addOwner(ReflectionProperty $owner): self
    {
        $this->owners[$owner->getName()] = $owner;

        return $this;
    }

    public function removeOwner(ReflectionProperty $owner): void
    {
        $this->owners->removeElement($owner);
    }

    /**
     * @return ArrayCollection<string, ReflectionProperty>
     */
    public function getOwners(): ArrayCollection
    {
        return $this->owners;
    }

    public function addProperty(string $prop): self
    {
        $this->properties[] = $prop;

        return $this;
    }

    public function removeProperty(string $prop): void
    {
        $this->properties->removeElement($prop);
    }

    /**
     * @return ArrayCollection<int, string>
     */
    public function getProperties(): ArrayCollection
    {
        return $this->properties;
    }

    /**
     * @param string $field
     *
     * @return string[]
     */
    public function getUserValues(string $field): ?array
    {
        return $this->userValues[$field] ?? null;
    }

    /**
     * @param string   $field
     * @param string[] $userValues
     *
     * @return AuditMetadata
     */
    public function setUserValues(string $field, array $userValues): self
    {
        $this->userValues[$field] = $userValues;

        return $this;
    }

    public function setEmbeddedClass(string $field, string $class): self
    {
        $this->embeddedFields[$field] = $class;

        return $this;
    }

    /**
     * Get the class name of the embedded class.
     *
     * @param string $field
     *
     * @return string
     */
    public function getEmbeddedClass(string $field): string
    {
        if (!isset($this->embeddedFields[$field])) {
            throw new InvalidArgumentException(\sprintf('Field "%s" is not marked as embedded.', $field));
        }

        return $this->embeddedFields[$field];
    }

    public function isEmbedded(string $field): bool
    {
        return isset($this->embeddedFields[$field]);
    }

    /**
     * @return ClassMetadata
     */
    public function getClassMetadata(): ClassMetadata
    {
        return $this->classMetadata;
    }

    public function trackSoftDelete(): void
    {
        $this->trackSoftDelete = true;
    }

    public function isTrackSoftDelete(): bool
    {
        return $this->trackSoftDelete;
    }

    /**
     * @return string
     */
    public function getAlias(): string
    {
        return $this->alias;
    }

    /**
     * @param string $alias
     *
     * @return AuditMetadata
     */
    public function setAlias(string $alias): self
    {
        $this->alias = $alias;

        return $this;
    }

    public function setDescriber(Describer $param): void
    {
        if (null !== $this->describer) {
            throw new LogicException('Only a single @Audit\Describe annotation allowed per class.');
        }

        $this->describer = $param;
    }

    public function getDescriber(): ?Describer
    {
        return $this->describer;
    }

    /**
     * @param mixed $object
     *
     * @return null|string
     */
    public function describe($object): ?string
    {
        if (null === $this->describer) {
            return null;
        }

        return $this->describer->describe($object);
    }

    public function canDescribe(): bool
    {
        return null !== $this->describer;
    }

    /**
     * Performs some extra configurations.
     *
     * Should be called after all annotation related data is set.
     */
    public function compile(): void
    {
        if (null === $this->describer && $this->classMetadata->getReflectionClass()->hasMethod('__toString')) {
            // if we don't have a describer set, attempt to use the simple to string describer
            $this->describer = new ToStringDescriber();
        }
    }
}
