<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Cyber\AuditBundle\Annotation\Apply;
use Cyber\AuditBundle\Annotation\Cascade;
use Cyber\AuditBundle\Annotation\Describe;
use Cyber\AuditBundle\Annotation\Skip;
use Cyber\AuditBundle\Annotation\Track;
use Cyber\AuditBundle\Describer\MethodDescriber;
use Cyber\AuditBundle\Describer\PropertyDescriber;
use Cyber\OrmExtras\Utility\SoftDeletable;
use Doctrine\Common\Annotations\Reader;
use Doctrine\ORM\Mapping\Embedded;
use Doctrine\Persistence\Mapping\ClassMetadata;
use InvalidArgumentException;
use ReflectionClass;

/**
 * @internal
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class AuditMetadataFactory
{
    /**
     * @var array<string, AuditMetadata|false>
     */
    private $auditMetadata = [];

    /** @var Reader */
    private $reader;

    public function __construct(Reader $reader)
    {
        $this->reader = $reader;
    }

    public function getAuditMetadata(ClassMetadata $classMetadata): ?AuditMetadata
    {
        $reflection = $classMetadata->getReflectionClass();
        $cacheKey   = $reflection->getName();
        if (isset($this->auditMetadata[$cacheKey])) {
            // convert false to null
            return $this->auditMetadata[$cacheKey] ?: null;
        }

        /** @var null|Apply $audit */
        $audit = $this->reader->getClassAnnotation($reflection, Apply::class);
        if (null === $audit) {
            // isset does not detect nulls so use false in our array
            $this->auditMetadata[$cacheKey] = false;

            return null;
        }

        $metadata = new AuditMetadata($classMetadata);
        $metadata->setAlias($audit->displayName ?? $reflection->getShortName());

        $this->processProperties($metadata, $reflection, Apply::INCLUDE_ALL === $audit->inclusionCriteria);

        foreach ($reflection->getMethods() as $method) {
            $annot = $this->reader->getMethodAnnotation($method, Describe::class);
            if (null !== $annot) {
                $metadata->setDescriber(new MethodDescriber($method->getName()));
            }
        }

        // check if entity is soft deleteable and add tracking flag to metadata
        if ($reflection->implementsInterface(SoftDeletable::class)) {
            $metadata->trackSoftDelete();
        }

        $metadata->compile();

        return $this->auditMetadata[$cacheKey] = $metadata;
    }

    /**
     * @param AuditMetadata           $metadata
     * @param ReflectionClass<object> $reflection
     * @param bool                    $all
     */
    private function processProperties(AuditMetadata $metadata, ReflectionClass $reflection, bool $all): void
    {
        foreach ($reflection->getProperties() as $property) {
            [$track, $skip] = $this->processProperty($metadata, $property);

            if ($track && $skip) {
                throw new InvalidArgumentException(\sprintf(
                    'You have set both Track and Skip annotation on %s::$%s, please pick only one.',
                    $reflection->getName(),
                    $property->getName()
                ));
            }

            if ($track || ($all && !$skip)) {
                $metadata->addProperty($property->getName());
            }
        }
    }

    /**
     * @param AuditMetadata       $metadata
     * @param \ReflectionProperty $property
     *
     * @return bool[]
     */
    private function processProperty(AuditMetadata $metadata, \ReflectionProperty $property): array
    {
        $track = false;
        $skip  = false;

        foreach ($this->reader->getPropertyAnnotations($property) as $annot) {
            switch (true) {
                case $annot instanceof Cascade:
                    $metadata->addOwner($property);
                    // TODO decide if should track cascade fields
                    // $metadata->removeProperty($property->getName());
                    break;
                case $annot instanceof Track:
                    $track = true;
                    if (null !== $annot->userValueList) {
                        $metadata->setUserValues($property->getName(), \constant($annot->userValueList));
                    }
                    break;
                case $annot instanceof Skip:
                    $skip = true;
                    break;
                case $annot instanceof Describe:
                    $metadata->setDescriber(new PropertyDescriber($property));
                    break;
                case $annot instanceof Embedded:
                    $metadata->setEmbeddedClass($property->getName(), $annot->class);
                    break;
            }
        }

        return [$track, $skip];
    }
}
