<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Service;

use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\Entity\Event;

/**
 * @internal
 */
class ChangeHydrator
{
    private $transformer;

    public function __construct(ChangeValueTransformer $transformer)
    {
        $this->transformer = $transformer;
    }

    /**
     * @param AuditBuilder                $builder
     * @param AuditMetadata               $metadata
     * @param array<string, array<mixed>> $changeSet keys are field names values are array with element 0 = old value
     *                                               and element 1 = new
     */
    public function fromChangeSet(AuditBuilder $builder, AuditMetadata $metadata, $changeSet): void
    {
        $trackedProperties = $metadata->getProperties();
        $owners            = $metadata->getOwners();
        foreach ($changeSet as $field => $valueChange) {
            if ($metadata->isTrackSoftDelete() && 'dateDeleted' === $field && null !== $valueChange[1]) {
                // TODO we need a more reliable way to get field name
                $builder->ofType(Event::TYPE_SOFT_DELETE);
            }

            if ($valueChange[0] === $valueChange[1]) {
                // nothing changed, technically it should not even be in changeSet but there is one special case:
                // on entity creation when nullable field is left blank null, both new and old values are null
                continue;
            }

            if (isset($owners[$field])) {
                // if owner's field changes we will need to track it under both entities
                $builder->ownerChange((string) $field, $valueChange);
            }

            if (!$trackedProperties->contains($field)) {
                continue;
            }

            if ($metadata->isEmbedded($field)) {
                $this->handleEmbeddable($field, $valueChange, $metadata, $builder);
                continue;
            }

            $oldValue = $this->transformer->transformField($field, $valueChange[0], $metadata);
            $newValue = $this->transformer->transformField($field, $valueChange[1], $metadata);

            $builder->changed(
                $field,
                $newValue->getRawValue(),
                $oldValue->getRawValue(),
                $newValue->getUserValue(),
                $oldValue->getUserValue()
            );
        }
    }

    /**
     * @param string        $field
     * @param array<mixed>  $valueChange
     * @param AuditMetadata $metadata
     * @param AuditBuilder  $builder
     */
    private function handleEmbeddable(
        string $field,
        array $valueChange,
        AuditMetadata $metadata,
        AuditBuilder $builder
    ): void {
        $oldValues  = $this->transformer->transformEmbedded($field, $valueChange[0], $metadata);
        $newValues  = $this->transformer->transformEmbedded($field, $valueChange[1], $metadata);
        $fieldPaths = \array_unique(\array_merge(\array_keys($oldValues), \array_keys($newValues)));
        foreach ($fieldPaths as $path) {
            // the path should always exist in both array, but just in case we check that it's there
            $builder->changed(
                $path,
                isset($newValues[$path]) ? $newValues[$path]->getRawValue() : null,
                isset($oldValues[$path]) ? $oldValues[$path]->getRawValue() : null,
                isset($newValues[$path]) ? $newValues[$path]->getUserValue() : null,
                isset($oldValues[$path]) ? $oldValues[$path]->getUserValue() : null
            );
        }
    }

    /**
     * Creates change entities from just original data.
     *
     * During delete operations doctrine does not provide a changeSet object. To create Change records for the audit
     * pass the original object data array to this function.
     *
     * @param AuditBuilder $builder
     * @param string[]     $original array of original values with keys being field names
     */
    public function fromOriginal(AuditBuilder $builder, AuditMetadata $metadata, $original): void
    {
        // convert the $original to a change set array and pass it to fromChangeset
        $changeSet = \array_map(function ($value) {
            return [$value, null];
        }, $original);

        $this->fromChangeSet($builder, $metadata, $changeSet);
    }
}
