<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\Service;

use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\Describer\ToStringDescriber;
use Cyber\AuditBundle\Entity\ChangeValue;
use Cyber\AuditBundle\Service\AuditManager;
use Cyber\AuditBundle\Service\ChangeValueTransformer;
use DateTime;
use DateTimeImmutable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\Persistence\ManagerRegistry;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Tests\Cyber\AuditBundle\Mock\MockComment;
use Tests\Cyber\AuditBundle\Mock\MockUser;
use Tests\Cyber\AuditBundle\Mock\MockUserTarget;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class ChangeValueTransformerTest extends TestCase
{
    /** @var ChangeValueTransformer */
    private $transformer;

    /** @var ManagerRegistry|MockObject */
    private $registry;

    /** @var AuditManager|MockObject */
    private $auditManager;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        $this->registry     = $this->getMockBuilder(ManagerRegistry::class)->getMock();
        $this->auditManager = $this->getMockBuilder(AuditManager::class)->disableOriginalConstructor()->getMock();
        $this->transformer  = new ChangeValueTransformer($this->registry, $this->auditManager);
    }

    /**
     * @dataProvider scalarData
     *
     * @param mixed $field
     * @param mixed $value
     * @param mixed $expected
     */
    public function testTransformScalar($field, $value, $expected): void
    {
        /** @var AuditMetadata|MockObject $auditMeta */
        $auditMeta = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();

        $result      = $this->transformer->transformField($field, $value, $auditMeta);
        $expectedObj = new ChangeValue($expected, $expected);
        $this->assertEquals($expectedObj, $result);
    }

    /**
     * @throws \Exception
     *
     * @return array<array<mixed>>
     */
    public function scalarData(): array
    {
        return [
            ['string', 'simple string', 'simple string'],
            ['dateonly', new DateTime('2018-05-27 00:00:00'), '2018-05-27'],
            ['datetime', new DateTimeImmutable('2018-05-27 01:00:00'), '2018-05-27 01:00:00'],
            ['int', 5, '5'],
            ['float', 5.5, '5.5'],
            ['bool', false, 'false'],
            ['null', null, '(null)'],
        ];
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testTransformScalarObject(): void
    {
        $this->transformer->transformScalar((object) ['someField' => 'name']);
    }

    /**
     * @dataProvider entityProvider
     *
     * @param mixed $entity
     * @param mixed $expectRaw
     * @param mixed $expectUser
     */
    public function testTransformEntity($entity, $expectRaw, $expectUser): void
    {
        /** @var ClassMetadata|MockObject $metadata */
        $metadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();
        /** @var AuditMetadata|MockObject $auditMeta */
        $auditMeta = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();

        $this->auditManager->expects($this->once())
            ->method('getEntityIdClosure')
            ->with($entity, $metadata)
            ->willReturnCallback(function (MockUser $entity) {
                return function () use ($entity) {
                    return $entity->getId();
                };
            });

        $this->auditManager->expects($this->any())
            ->method('getAuditMetadata')
            ->with($metadata)
            ->willReturn($auditMeta);
        $auditMeta->expects($this->any())
            ->method('describe')
            ->willReturnCallback(function ($value) {
                return (new ToStringDescriber())->describe($value);
            });

        $result = $this->transformer->transformEntity($entity, $metadata);

        $raw = $result->getRawValue();

        $this->assertInternalType('callable', $raw, '$raw not callable');

        $this->assertEquals($expectRaw, $raw());
        $this->assertEquals($expectUser, $result->getUserValue());
    }

    /**
     * @return array<array<mixed>>
     */
    public function entityProvider(): array
    {
        return [
            [
                (new MockUser())->setName('test')->setId(1),
                1,
                'test',
            ],
            [
                (new MockUserTarget())->setName('test')->setId(1),
                1,
                'test',
            ],
        ];
    }

    public function testTransformScalarField(): void
    {
        /** @var AuditMetadata|MockObject $metadata */
        $metadata = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();
        $metadata->expects($this->any())
            ->method('getUserValues')
            ->withConsecutive(['id'], ['emails'])
            ->willReturnOnConsecutiveCalls([1 => 'root'], null);

        $result = $this->transformer->transformField('id', 1, $metadata);

        $this->assertEquals(1, $result->getRawValue());
        $this->assertEquals('root', $result->getUserValue());

        $result = $this->transformer->transformField('emails', ['a@a.com', 'b@b.com'], $metadata);

        $this->assertEquals(['a@a.com', 'b@b.com'], $result->getRawValue());
        $this->assertEquals(['a@a.com', 'b@b.com'], $result->getUserValue());
    }

    public function testTransformSimpleObjects(): void
    {
        /** @var AuditMetadata|MockObject $metadata */
        $metadata = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();

        $comments = new ArrayCollection();
        $comments->add((new MockComment())->setId(1)->setComment('com1'));
        $comments->add((new MockComment())->setId(2)->setComment('com2'));

        $result = $this->transformer->transformField('comments', $comments, $metadata);

        $rawValues = $result->getRawValue();

        $this->assertEquals(['com1', 'com2'], $rawValues);
        $this->assertEquals(['com1', 'com2'], $result->getUserValue());
    }

    public function testTransformCollectionField(): void
    {
        /** @var AuditMetadata|MockObject $metadata */
        $metadata = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();
        /** @var ClassMetadata|MockObject $classMetadata */
        $classMetadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();
        /** @var ClassMetadata|MockObject $childMetadata */
        $childMetadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();

        /** @var EntityManagerInterface|MockObject $manager */
        $manager = $this->getMockBuilder(EntityManagerInterface::class)->getMock();

        $metadata->expects($this->any())
            ->method('getClassMetadata')
            ->willReturn($classMetadata);

        $classMetadata->expects($this->any())
            ->method('hasAssociation')
            ->with('comments')
            ->willReturn(true);

        $classMetadata->expects($this->any())
            ->method('getAssociationTargetClass')
            ->with('comments')
            ->willReturn(MockComment::class);

        $this->registry->expects($this->any())
            ->method('getManagerForClass')
            ->with(MockComment::class)
            ->willReturn($manager);

        $manager->expects($this->any())
            ->method('getClassMetadata')
            ->with(MockComment::class)
            ->willReturn($childMetadata);

        $this->auditManager->expects($this->any())
            ->method('getEntityIdClosure')
            ->willReturnCallback(function (MockComment $comment) {
                return function () use ($comment) {
                    return $comment->getId();
                };
            });

        // technically this should be different metadata object as it is for different entity, but for purposes of this tests this will suffice
        $this->auditManager->expects($this->any())
            ->method('getAuditMetadata')
            ->with($childMetadata)
            ->willReturn($metadata);
        $metadata->expects($this->any())
            ->method('describe')
            ->willReturnCallback(function ($value) {
                return (new ToStringDescriber())->describe($value);
            });

        $comments = new ArrayCollection();
        $comments->add((new MockComment())->setId(1)->setComment('com1'));
        $comments->add((new MockComment())->setId(2)->setComment('com2'));

        $result = $this->transformer->transformField('comments', $comments, $metadata);

        $rawValues = (array) $result->getRawValue();

        $this->assertCount(2, $rawValues);

        $resolved = \array_map(function ($callback) {
            return $callback();
        }, $rawValues);

        $this->assertEquals(['1', '2'], $resolved);
        $this->assertEquals(['com1', 'com2'], $result->getUserValue());
    }
}
