<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\Service;

use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Cyber\AuditBundle\Service\EntityFactory;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

class AuditBuilderTest extends TestCase
{
    /** @var AuditBuilder */
    private $builder;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        /** @var EntityFactory|MockObject $factory */
        $factory = $this->getMockBuilder(EntityFactory::class)->disableOriginalConstructor()->getMock();

        $factory->expects($this->any())
            ->method('createEvent')
            ->willReturnCallback(function () {
                return new class() extends Event {
                };
            });

        $factory->expects($this->any())
            ->method('createChange')
            ->willReturnCallback(function () {
                return new class() extends Change {
                };
            });

        $factory->expects($this->any())
            ->method('createMap')
            ->willReturnCallback(function () {
                return new class() extends EventMap {
                };
            });

        $this->builder = new AuditBuilder($factory);
    }

    /**
     * @param string $call
     * @param int    $type
     *
     * @dataProvider eventTypeProvider
     */
    public function testBuildCreateEvent($call, $type): void
    {
        /** @var \Cyber\AuditBundle\AuditBuilder $builder */
        $builder = $this->builder->$call('Some\FQCN');

        /** @var Event $event */
        $event = $builder
            ->withId('7')
            ->alias('Patient')
            ->description('John Smith')
            ->changed('user', '77')
            ->mappedTo('Another\FQCN', '44')
            ->message('Good day')
            ->getEvent();

        $this->assertInstanceOf(Event::class, $event);

        $this->assertEquals($type, $event->getType());
        $this->assertEquals('Some\FQCN', $event->getEntityClass());
        $this->assertEquals('Patient', $event->getEntityAlias());
        $this->assertEquals(7, $event->getEntityId());
        $this->assertEquals('John Smith', $event->getDescription());
        $this->assertEquals('Good day', $event->getMessage());

        $changes = $event->getChanges();
        $this->assertCount(1, $changes);
        /** @var Change $change */
        $change = $changes[0];
        $this->assertEquals('user', $change->getField());
        $this->assertEquals('77', $change->getNewValue()->getRawValue());

        $maps = $event->getMaps();
        $this->assertCount(1, $maps);
        /** @var EventMap $map */
        $map = $maps[0];
        $this->assertEquals('Another\FQCN', $map->getEntityClass());
        $this->assertEquals(44, $map->getEntityId());
    }

    /**
     * @return \Generator<array<mixed>>
     */
    public function eventTypeProvider(): \Generator
    {
        yield ['created', Event::TYPE_INSERT];
        yield ['updated', Event::TYPE_UPDATE];
        yield ['deleted', Event::TYPE_DELETE];
        yield ['softDeleted', Event::TYPE_SOFT_DELETE];
        yield ['messageFor', Event::TYPE_MESSAGE];
    }

    /**
     * @group        debug
     */
    public function testValidateValidBuilder(): void
    {
        $result = $this->builder->created('someclass')
            ->alias('somealias')
            ->withId('someid')
            ->mappedTo('someclass', 'someid')
            ->validate();

        $this->assertSame($this->builder, $result);
    }

    /**
     * @expectedException \LogicException
     * @dataProvider invalidBuilderData
     *
     * @param array<string, array> $data
     */
    public function testValidateInvalidBuilder($data): void
    {
        foreach ($data as $method => $arguments) {
            $callable = [$this->builder, $method];
            $this->assertInternalType('callable', $callable);
            \call_user_func_array($callable, $arguments);
        }
        $this->builder->validate();
        $this->builder->validateMapping();
    }

    /**
     * @return \Generator<array<mixed>>
     */
    public function invalidBuilderData(): \Generator
    {
        yield [[]];
        yield [['ofType' => [0]]];
        yield [['created' => ['classname']]];
        yield [
            [
                'created' => ['classname'],
                'alias'   => ['somealias'],
            ],
        ];
        yield [
            [
                'created' => ['classname'],
                'alias'   => ['somealias'],
                'withId'  => ['someid'],
            ],
        ];
    }
}
