<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Entity;

use Cyber\OrmExtras\Utility\CreatorAttributes;
use Cyber\OrmExtras\Utility\CreatorAttributeTrait;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\MappedSuperclass
 */
class Event implements CreatorAttributes
{
    use CreatorAttributeTrait;

    const TYPE_INSERT = 0;

    const TYPE_UPDATE = 25;

    const TYPE_DELETE = 50;

    const TYPE_SOFT_DELETE = 75;

    const TYPE_MESSAGE = 100;

    const TYPE_NAMES = [
        self::TYPE_INSERT      => 'Created',
        self::TYPE_UPDATE      => 'Updated',
        self::TYPE_DELETE      => 'Deleted',
        self::TYPE_SOFT_DELETE => 'Soft Deleted',
        self::TYPE_MESSAGE     => 'Message',
    ];

    /**
     * @var \DateTime
     * @ORM\Column(type="datetime")
     */
    protected $dateCreated;

    /**
     * @var ArrayCollection<int, Change>|Change[]
     */
    protected $changes;

    /**
     * @var ArrayCollection<int, EventMap>|EventMap[]
     */
    protected $maps;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=255, nullable=false)
     */
    protected $entityClass;

    /**
     * @var null|callable|callable[]|string|string[]
     *
     * @ORM\Column(type="integer", nullable=false)
     */
    protected $entityId;

    /**
     * A user friendly entity name.
     *
     * @var string
     *
     * @ORM\Column(type="string", length=80, nullable=false)
     */
    protected $entityAlias;

    /**
     * A user friendly target name.
     *
     * @var null|string
     *
     * @ORM\Column(type="string", length=255, nullable=true)
     */
    protected $description;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", nullable=false)
     */
    protected $type;

    /**
     * @var null|string
     *
     * @ORM\Column(type="text", nullable=true)
     */
    protected $message;

    public function __construct()
    {
        $this->maps    = new ArrayCollection();
        $this->changes = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getEntityClass(): ?string
    {
        return $this->entityClass;
    }

    /**
     * @param string $entityClass
     *
     * @return Event
     */
    public function setEntityClass(string $entityClass): self
    {
        $this->entityClass = $entityClass;

        return $this;
    }

    /**
     * @return null|callable|callable[]|string|string[]
     */
    public function getEntityId()
    {
        return $this->entityId;
    }

    /**
     * @param null|callable|callable[]|string|string[] $entityId
     *
     * @return Event
     */
    public function setEntityId($entityId): self
    {
        $this->entityId = $entityId;

        return $this;
    }

    /**
     * @return string
     */
    public function getEntityAlias(): ?string
    {
        return $this->entityAlias;
    }

    /**
     * @param string $entityAlias
     *
     * @return Event
     */
    public function setEntityAlias(string $entityAlias): self
    {
        $this->entityAlias = $entityAlias;

        return $this;
    }

    /**
     * @return int
     */
    public function getType(): ?int
    {
        return $this->type;
    }

    /**
     * Type of event that occurred (ex. Created).
     *
     * @return string
     */
    public function getTypeLabel(): ?string
    {
        return self::TYPE_NAMES[$this->type] ?? 'None';
    }

    /**
     * @param int $type
     *
     * @return Event
     */
    public function setType(int $type): self
    {
        $this->type = $type;

        return $this;
    }

    /**
     * @return ArrayCollection<int,Change>|Change[]
     */
    public function getChanges()
    {
        return $this->changes;
    }

    /**
     * @param Change $change
     *
     * @return Event
     */
    public function addChange(Change $change): self
    {
        $change->setEvent($this);
        $this->changes[] = $change;

        return $this;
    }

    public function removeChange(Change $change): void
    {
        $this->changes->removeElement($change);
    }

    /**
     * @return string
     */
    public function getMessage(): ?string
    {
        return $this->message;
    }

    /**
     * @param string $message
     *
     * @return Event
     */
    public function setMessage(?string $message): self
    {
        $this->message = $message;

        return $this;
    }

    public function addMap(EventMap $map): self
    {
        $map->setEvent($this);
        $this->maps[] = $map;

        return $this;
    }

    public function removeMap(EventMap $map): void
    {
        $this->maps->removeElement($map);
    }

    /**
     * @return ArrayCollection<int,EventMap>|EventMap[]
     */
    public function getMaps()
    {
        return $this->maps;
    }

    /**
     * @return null|string
     */
    public function getDescription(): ?string
    {
        return $this->description;
    }

    /**
     * @param string $description
     *
     * @return Event
     */
    public function setDescription(?string $description): self
    {
        $this->description = $description;

        return $this;
    }
}
