<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\EventListener;

use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Cyber\AuditBundle\EventListener\AuditDoctrineSubscriber;
use Cyber\AuditBundle\Service\AuditManager;
use Cyber\AuditBundle\Service\ChangeHydrator;
use Cyber\AuditBundle\Service\EntityFactory;
use Doctrine\Common\Annotations\Reader;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\ORM\Events;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\UnitOfWork;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.ExcessiveClassLength)
 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
 */
class AuditDoctrineSubscriberTest extends TestCase
{
    /**
     * Should process entity and add created event to queue.
     *
     * @throws \ReflectionException
     */
    public function testShouldProcessEntityAndAddCreatedEventToQueue(): void
    {
        $testStorage      = new class() {
            /**
             * @var array<mixed>
             */
            private $data = [];

            /**
             * @param mixed $data
             */
            public function setData($data): void
            {
                $this->data[] = $data;
            }

            /**
             * @return mixed
             */
            public function getData()
            {
                return $this->data;
            }
        };

        /** @var MockObject|Reader $reader */
        $reader = $this->getMockBuilder(Reader::class)
            ->disableOriginalConstructor()
            ->getMock();

        /** @var AuditManager|MockObject $auditManager */
        $auditManager = $this->getMockBuilder(AuditManager::class)
            ->disableOriginalConstructor()
            ->getMock();

        $auditManager->method('enqueueAudit')
            ->willReturnCallback(function ($event) use ($testStorage) {
                $testStorage->setData($event);
            });

        /** @var ClassMetadata<object>|MockObject $metadata */
        $metadata = $this->getMockBuilder(ClassMetadata::class)
            ->disableOriginalConstructor()
            ->getMock();

        $factory = new EntityFactory(Change::class, Event::class, EventMap::class);

        $auditManager->method('createAuditBuilder')
            ->willReturnCallback(function () use ($factory) {
                return new AuditBuilder($factory);
            });

        /** @var AuditMetadata<object>|MockObject $auditMetadata */
        $auditMetadata = $this->getMockBuilder(AuditMetadata::class)
            ->disableOriginalConstructor()
            ->getMock();

        $auditMetadata->method('getOwners')
            ->willReturn([]);

        $auditMetadata->method('getClassMetadata')
            ->willReturn($metadata);

        $auditMetadata->method('getAlias')
            ->willReturn('Patient');

        $auditManager->method('getAuditMetadata')
            ->willReturn($auditMetadata);

        /** @var ChangeHydrator|MockObject $hydrator */
        $hydrator = $this->getMockBuilder(ChangeHydrator::class)
            ->disableOriginalConstructor()
            ->getMock();

        $events = [
            Events::onFlush,
        ];

        $subscriber = new AuditDoctrineSubscriber($reader, $hydrator, $auditManager);

        $this->assertEquals($events, $subscriber->getSubscribedEvents());

        /** @var MockObject|OnFlushEventArgs $eventArgs */
        $eventArgs = $this->getMockBuilder(OnFlushEventArgs::class)
            ->disableOriginalConstructor()
            ->getMock();

        $manager = $this->getMockBuilder(EntityManagerInterface::class)
            ->getMock();

        $reflection = $this->getMockBuilder(\ReflectionClass::class)
            ->disableOriginalConstructor()
            ->getMock();

        $reflection->method('getName')
            ->willReturn('Patient');

        $metadata->method('getReflectionClass')
            ->willReturn($reflection);

        $uow = $this->getUnitOfWork();

        $manager->method('getUnitOfWork')
            ->willReturn($uow);

        $manager->method('getClassMetadata')
            ->willReturn($metadata);

        $eventArgs->method('getEntityManager')
            ->willReturn($manager);

        $subscriber->onFlush($eventArgs);

        $data = $testStorage->getData();

        $this->assertCount(4, $data);
        foreach ($data as $item) {
            $this->assertInstanceOf(AuditBuilder::class, $item);
        }
    }

    private function getUnitOfWork(): UnitOfWork
    {
        $patient = new class() {
            public function getTargetName(): string
            {
                return 'Name';
            }
        };

        $physician = new class() {
        };

        $entities = [
            $physician,
            $patient,
        ];

        /** @var MockObject|UnitOfWork $uow */
        $uow = $this->getMockBuilder(UnitOfWork::class)
            ->disableOriginalConstructor()
            ->getMock();

        $uow->method('getScheduledEntityInsertions')
            ->willReturn($entities);

        $uow->method('getScheduledEntityUpdates')
            ->willReturn([$entities[1]]);

        $uow->method('getScheduledEntityDeletions')
            ->willReturn([$entities[1]]);

        $uow->method('getScheduledCollectionUpdates')
            ->willReturn([]);

        return $uow;
    }
}
