<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\ChangeValue;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Cyber\AuditBundle\Service\EntityFactory;
use Doctrine\Common\Collections\Collection;
use LogicException;

/**
 * @SuppressWarnings(PHPMD.TooManyPublicMethods)
 */
class AuditBuilder
{
    /** @var EntityFactory */
    private $factory;

    /** @var int */
    private $eventType;

    /** @var string */
    private $entityClass;

    /** @var string */
    private $alias;

    /** @var null|callable|callable[]|string|string[] */
    private $id;

    /** @var null|string */
    private $description;

    /** @var null|string */
    private $message;

    /** @var Change[] */
    private $changes = [];

    /** @var EventMap[] */
    private $maps = [];

    /** @var array<string, array<int, mixed>|Collection<mixed, mixed>> */
    private $ownerChanges = [];

    /** @var null|object */
    private $entityObject;

    /**
     * @param EntityFactory $factory
     * @param null|object   $entityObject
     */
    public function __construct(EntityFactory $factory, $entityObject = null)
    {
        $this->factory      = $factory;
        $this->entityObject = $entityObject;
    }

    public function created(string $class): self
    {
        return $this->eventFor(Event::TYPE_INSERT, $class);
    }

    public function updated(string $class): self
    {
        return $this->eventFor(Event::TYPE_UPDATE, $class);
    }

    public function deleted(string $class): self
    {
        return $this->eventFor(Event::TYPE_DELETE, $class);
    }

    public function softDeleted(string $class): self
    {
        return $this->eventFor(Event::TYPE_SOFT_DELETE, $class);
    }

    public function messageFor(string $class): self
    {
        return $this->eventFor(Event::TYPE_MESSAGE, $class);
    }

    public function eventFor(int $type, string $class): self
    {
        $this->entityClass = $class;
        $this->eventType   = $type;

        return $this;
    }

    /**
     * Let's you override the event type for this audit.
     *
     * For example it may have started as UPDATE, but what updated was soft delete field
     * so you would need to change the type to SOFT_DELETED
     *
     * @param int $type
     *
     * @return AuditBuilder
     */
    public function ofType(int $type): self
    {
        $this->eventType = $type;

        return $this;
    }

    /**
     * @param null|callable|callable[]|string|string[] $id
     *
     * @return $this
     */
    public function withId($id): self
    {
        $this->id = $id;

        return $this;
    }

    public function alias(string $alias): self
    {
        $this->alias = $alias;

        return $this;
    }

    public function description(?string $description): self
    {
        $this->description = $description;

        return $this;
    }

    public function message(string $message): self
    {
        $this->message = $message;

        return $this;
    }

    /**
     * @param string                                   $field
     * @param null|callable|callable[]|string|string[] $newValue
     * @param null|callable|callable[]|string|string[] $oldValue
     * @param null|callable|callable[]|string|string[] $newUserValue
     * @param null|callable|callable[]|string|string[] $oldUserValue
     *
     * @return $this
     */
    public function changed(
        string $field,
        $newValue,
        $oldValue = null,
        $newUserValue = null,
        $oldUserValue = null
    ): self {
        $oldChangeVal = new ChangeValue($oldValue, $oldUserValue);
        $newChangeVal = new ChangeValue($newValue, $newUserValue);

        $change = $this->factory->createChange();
        $change->setField($field)
            ->setNewValue($newChangeVal)
            ->setOldValue($oldChangeVal);

        $this->changes[] = $change;

        return $this;
    }

    /**
     * @param string          $class
     * @param callable|string $id
     *
     * @return $this
     */
    public function mappedTo(string $class, $id): self
    {
        $map = $this->factory->createMap();

        $map->setEntityClass($class)
            ->setEntityId($id);

        $this->maps[] = $map;

        return $this;
    }

    public function getEvent(): Event
    {
        $event = $this->factory->createEvent();

        $event->setType($this->eventType)
            ->setEntityClass($this->entityClass)
            ->setEntityAlias($this->alias)
            ->setEntityId($this->id)
            ->setDescription($this->description)
            ->setMessage($this->message);

        foreach ($this->changes as $change) {
            $event->addChange($change);
        }

        foreach ($this->maps as $map) {
            $event->addMap($map);
        }

        return $event;
    }

    public function getResolvedEvent(): Event
    {
        $event = $this->factory->createEvent();

        $event->setType($this->eventType)
            ->setEntityClass($this->entityClass)
            ->setEntityAlias($this->alias)
            ->setEntityId($this->resolveValue($this->id))
            ->setDescription($this->description)
            ->setMessage($this->message);

        foreach ($this->changes as $change) {
            $oldValue = $change->getOldValue();
            $newValue = $change->getNewValue();

            $oldValue->setRawValue($this->resolveValue($oldValue->getRawValue()));
            $oldValue->setUserValue($this->resolveValue($oldValue->getUserValue()));
            $newValue->setRawValue($this->resolveValue($newValue->getRawValue()));
            $newValue->setUserValue($this->resolveValue($newValue->getUserValue()));

            $event->addChange($change);
        }

        foreach ($this->maps as $map) {
            $map->setEntityId($this->resolveValue($map->getEntityId()));
            $event->addMap($map);
        }

        return $event;
    }

    public function isValid(): bool
    {
        switch ($this->eventType) {
            case Event::TYPE_MESSAGE:
                // if message with empty message it is not valid
                return !empty($this->message);
            case Event::TYPE_SOFT_DELETE:
                // no special requirements for soft deletion
                return true;
            default:
                // if change with no changes it is invalid
                return isset($this->changes[0]);
        }
    }

    public function getEntityClass(): string
    {
        return $this->entityClass ?? '';
    }

    public function validate(): self
    {
        switch (true) {
            case null === $this->eventType:
                throw $this->invalidException(
                    'event type',
                    ['created()', 'updated()', 'deleted()', 'softDeleted()', 'messageFor()', 'eventFor()']
                );
            case null === $this->entityClass:
                throw $this->invalidException(
                    'entity class',
                    ['created()', 'updated()', 'deleted()', 'softDeleted()', 'messageFor()', 'eventFor()']
                );
            case null === $this->alias:
                throw $this->invalidException('alias', ['alias()']);
            case null === $this->id:
                throw $this->invalidException('id', ['withId()']);
        }

        return $this;
    }

    public function validateMapping(): void
    {
        if (!isset($this->maps[0])) {
            throw $this->invalidException('maps', ['mappedTo() at least once (for self mapping)']);
        }
    }

    /**
     * @param string                                     $field
     * @param array<int, mixed>|Collection<mixed, mixed> $valueChange
     */
    public function ownerChange(string $field, $valueChange): void
    {
        $this->ownerChanges[$field] = $valueChange;
    }

    /**
     * @param string $field
     *
     * @return null|array<int, mixed>|Collection<mixed, mixed>
     */
    public function findOwnerChange(string $field)
    {
        return $this->ownerChanges[$field] ?? null;
    }

    /**
     * @return null|object
     */
    public function getEntityObject()
    {
        return $this->entityObject;
    }

    /**
     * @param string   $key
     * @param string[] $methods
     *
     * @return LogicException
     */
    private function invalidException(string $key, array $methods): LogicException
    {
        $message = \sprintf('Missing %s, you must call one of: %s', $key, \implode(', ', $methods));

        return new LogicException($message);
    }

    /**
     * If value is callable returns the result of the call, otherwise returns value.
     *
     * @param null|callable|callable[]|string|string[] $value
     *
     * @return mixed
     */
    private function resolveValue($value)
    {
        if ($value instanceof \Closure) {
            $value = $value();
        }

        if (\is_array($value)) {
            return \json_encode(\array_map([$this, 'resolveValue'], $value));
        }

        return $value;
    }
}
