<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Service;

use Cyber\AuditBundle\Annotation\Apply;
use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\ChangeValue;
use Doctrine\Persistence\ManagerRegistry;
use Doctrine\Persistence\Mapping\ClassMetadata;
use Doctrine\Persistence\ObjectManager;
use InvalidArgumentException;
use ReflectionException;
use RuntimeException;

/**
 * @internal
 */
class ChangeValueTransformer
{
    const ZERO_TIME = '00:00:00';

    private $registry;

    private $manager;

    public function __construct(ManagerRegistry $registry, AuditManager $manager)
    {
        $this->registry = $registry;
        $this->manager  = $manager;
    }

    /**
     * @param mixed $value
     *
     * @return string
     */
    public function transformScalar($value): string
    {
        if (\is_string($value)) {
            return $value;
        }

        if (\is_bool($value)) {
            return $value ? 'true' : 'false';
        }

        if ($value instanceof \DateTimeInterface) {
            return self::ZERO_TIME === $value->format('H:i:s') ? $value->format('Y-m-d') : $value->format('Y-m-d H:i:s');
        }

        if (null === $value) {
            return Change::NULL_CHANGE;
        }

        try {
            return (string) $value;
        } catch (\Throwable $e) {
            throw new InvalidArgumentException('The $value must be scalar');
        }
    }

    /**
     * @param object                $entity
     * @param ClassMetadata<object> $classMetadata
     *
     * @return ChangeValue
     */
    public function transformEntity($entity, ClassMetadata $classMetadata): ChangeValue
    {
        $entityId      = $this->manager->getEntityIdClosure($entity, $classMetadata);
        $auditMetadata = $this->manager->getAuditMetadata($classMetadata);

        $targetName = $auditMetadata ? $auditMetadata->describe($entity) : null;

        return new ChangeValue($entityId, $targetName ?? $entityId);
    }

    /**
     * @param string                $fieldName
     * @param mixed                 $value
     * @param AuditMetadata<object> $auditMetadata
     *
     * @return ChangeValue
     */
    public function transformField(string $fieldName, $value, AuditMetadata $auditMetadata): ChangeValue
    {
        if (null === $value || \is_scalar($value) || $value instanceof \DateTimeInterface) {
            $userValues = $auditMetadata->getUserValues($fieldName);
            $value      = $this->transformScalar($value);

            return new ChangeValue($value, $userValues[$value] ?? $value);
        }

        if (\is_iterable($value)) {
            return $this->transformIterable($value, $fieldName, $auditMetadata);
        }

        $classMetadata = $auditMetadata->getClassMetadata();

        if (null !== $classMetadata && $classMetadata->hasAssociation($fieldName)) {
            /** @var class-string<object> $associationClass should be related entity, so convert entity */
            $associationClass = $classMetadata->getAssociationTargetClass($fieldName);
            /** @var ObjectManager $manager typehint that this will not be null */
            $manager = $this->registry->getManagerForClass($associationClass);

            return $this->transformEntity($value, $manager->getClassMetadata($associationClass));
        }

        // at this point we got an object which should have __toString implemented, otherwise it's configuration error
        return $this->tryTransformToString($value, $fieldName, $classMetadata);
    }

    /**
     * @param string                $fieldName
     * @param mixed                 $value
     * @param AuditMetadata<object> $parentMetadata
     *
     * @return ChangeValue[]
     */
    public function transformEmbedded(string $fieldName, $value, AuditMetadata $parentMetadata): array
    {
        $changes = [];
        $class   = $parentMetadata->getEmbeddedClass($fieldName);
        /** @var ObjectManager $manager cannot be null, embeddables don't have managers, so use parent class manager */
        $manager  = $this->registry->getManagerForClass($parentMetadata->getClassMetadata()->getName());
        $metadata = $manager->getClassMetadata($class);
        $audMeta  = $this->manager->getAuditMetadata($metadata);
        if (null === $audMeta) {
            throw new InvalidArgumentException(
                \sprintf(
                    'It seems like you requested to track Embedded property "%s", but the class associated with it "%s" was not registered with AuditManager, probably due to lack of "%s" annotation',
                    $fieldName,
                    $class,
                    Apply::class
                )
            );
        }
        $props = $audMeta->getProperties();

        foreach ($props as $childField) {
            try {
                $propRef = $metadata->getReflectionClass()->getProperty($childField);
            } catch (ReflectionException $ex) {
                throw new RuntimeException('Reflection failure', 0, $ex);
            }
            $propRef->setAccessible(true);
            $childValue = null === $value ? null : $propRef->getValue($value);

            if ($audMeta->isEmbedded($childField)) {
                $nestedChanges = $this->transformEmbedded($childField, $childValue, $audMeta);
                foreach ($nestedChanges as $propPath => $change) {
                    $changes[$fieldName . '.' . $propPath] = $change;
                }

                continue;
            }

            $changes[$fieldName . '.' . $childField] = $this->transformField($childField, $childValue, $audMeta);
        }

        return $changes;
    }

    /**
     * @param iterable<mixed>       $value
     * @param string                $fieldName
     * @param AuditMetadata<object> $auditMetadata
     *
     * @return ChangeValue
     */
    private function transformIterable(iterable $value, string $fieldName, AuditMetadata $auditMetadata): ChangeValue
    {
        $changes = [];
        foreach ($value as $val) {
            $changes[] = $this->transformField($fieldName, $val, $auditMetadata);
        }

        $rawValues = \array_reduce($changes, function ($carry, ChangeValue $item) {
            $carry[] = $item->getRawValue();

            return $carry;
        }, []);

        $userValues = \array_reduce($changes, function ($carry, ChangeValue $item) {
            $carry[] = $item->getUserValue();

            return $carry;
        }, []);

        return new ChangeValue($rawValues, $userValues);
    }

    /**
     * @param object                     $value
     * @param string                     $fieldName
     * @param null|ClassMetadata<object> $classMetadata
     *
     * @return ChangeValue
     */
    private function tryTransformToString($value, $fieldName, ?ClassMetadata $classMetadata): ChangeValue
    {
        if (!\method_exists($value, '__toString')) {
            throw new InvalidArgumentException(
                \sprintf(
                    'Value in field "%s" of class "%s" could not be converted to string.',
                    $fieldName,
                    $classMetadata ? $classMetadata->getName() : '[Unknown]'
                )
            );
        }

        $value = (string) $value;

        return new ChangeValue($value, $value);
    }
}
