<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\EventListener;

use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\AuditEvents;
use Cyber\AuditBundle\Service\AuditManager;
use Cyber\AuditBundle\Service\AuditQueue;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Doctrine\Persistence\ObjectManager;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\ConsoleEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;
use UnexpectedValueException;

class AuditKernelSubscriber implements EventSubscriberInterface
{
    /** @var ManagerRegistry */
    private $doctrine;

    /** @var null|LoggerInterface */
    private $logger;

    /** @var AuditQueue<AuditBuilder> */
    private $queue;

    private $manager;

    public function __construct(ManagerRegistry $doctrine, AuditQueue $queue, AuditManager $manager)
    {
        $this->doctrine = $doctrine;
        $this->queue    = $queue;
        $this->manager  = $manager;
    }

    /**
     * @param null|LoggerInterface $logger
     *
     * @required
     */
    public function setLogger(LoggerInterface $logger = null): void
    {
        $this->logger = $logger;
    }

    /**
     * @return string[]
     */
    public static function getSubscribedEvents(): array
    {
        return [
            KernelEvents::TERMINATE  => 'storeAudits',
            ConsoleEvents::TERMINATE => 'storeAudits',
            AuditEvents::FLUSH_QUEUE => 'storeAudits',
        ];
    }

    public function storeAudits(): void
    {
        if ($this->queue->isEmpty()) {
            return;
        }

        $eventClass = \get_class($this->queue->top()->getEvent());
        $em         = $this->getEntityManager($eventClass);

        if (null === $em) {
            return;
        }

        foreach ($this->queue as $builder) {
            if (!$builder->isValid()) {
                // if builder not valid no reason to waste db space with invalid data
                continue;
            }

            if ($this->manager->isDisableForEntity($builder->getEntityClass())) {
                continue;
            }

            $builder->validateMapping();
            $event = $builder->getResolvedEvent();

            $em->persist($event);
        }

        try {
            $em->flush();
        } catch (\Exception $e) {
            if ($this->logger) {
                $this->logger->warning('Failed to record audit log: ' . $e->getMessage(), ['exception' => $e]);
            }
        }
    }

    private function getEntityManager(string $eventClass): ?ObjectManager
    {
        $em = $this->doctrine->getManagerForClass($eventClass);

        if (null === $em) {
            throw new UnexpectedValueException(\sprintf('Could not find entity manger for %s entity', $eventClass));
        }

        if ($this->logger && $em instanceof EntityManagerInterface && !$em->isOpen()) {
            $this->logger->warning(
                'Unable to record audits: ObjectManager is closed. Probably some exception occurred during transaction.'
            );

            return null;
        }

        return $em;
    }
}
