<?php

declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Service;

use Attribute;
use Cyber\AuditBundle\Annotation\Annotation;
use Doctrine\Common\Annotations\Reader;
use Doctrine\ORM\Mapping\Driver\RepeatableAttributeCollection;
use LogicException;
use ReflectionAttribute;
use ReflectionClass;
use ReflectionMethod;
use ReflectionProperty;

/**
 * @internal
 *
 * @SuppressWarnings(PHPMD) borrowed from Doctrine
 */
final class AttributeReader implements Reader
{
    /** @var array<class-string<Annotation>,bool> */
    private array $isRepeatableAttribute = [];

    /**
     * @template T of Annotation
     *
     * @return T[]
     */
    public function getClassAnnotations(ReflectionClass $class): array
    {
        return $this->convertToAttributeInstances($class->getAttributes());
    }

    /**
     * @template T of Annotation
     */
    public function getMethodAnnotations(ReflectionMethod $method): array
    {
        return $this->convertToAttributeInstances($method->getAttributes());
    }

    /**
     * @template T of Annotation
     */
    public function getPropertyAnnotations(ReflectionProperty $property): array
    {
        return $this->convertToAttributeInstances($property->getAttributes());
    }

    /**
     * @param class-string<T> $annotationName the name of the annotation
     *
     * @return null|T
     *
     * @template T of Annotation
     */
    public function getPropertyAnnotation(ReflectionProperty $property, $annotationName)
    {
        if ($this->isRepeatable($annotationName)) {
            throw new LogicException(\sprintf(
                'The attribute "%s" is repeatable. Call getPropertyAnnotationCollection() instead.',
                $annotationName
            ));
        }

        return $this->getPropertyAnnotations($property)[$annotationName]
            ?? ($this->isRepeatable($annotationName) ? new RepeatableAttributeCollection() : null);
    }

    /**
     * @param ReflectionClass<object> $class
     * @param class-string            $annotationName
     *
     * @return null|mixed
     */
    public function getClassAnnotation(ReflectionClass $class, $annotationName)
    {
        $annotations = $this->getClassAnnotations($class);

        foreach ($annotations as $annotation) {
            if ($annotation instanceof $annotationName) {
                return $annotation;
            }
        }

        return null;
    }

    public function getMethodAnnotation(ReflectionMethod $method, $annotationName)
    {
        $annotations = $this->getMethodAnnotations($method);

        foreach ($annotations as $annotation) {
            if ($annotation instanceof $annotationName) {
                return $annotation;
            }
        }

        return null;
    }

    /**
     * @param class-string<T> $annotationName the name of the annotation
     *
     * @return RepeatableAttributeCollection<T>
     *
     * @template T of Annotation
     */
    public function getPropertyAnnotationCollection(
        ReflectionProperty $property,
        string $annotationName
    ): RepeatableAttributeCollection {
        if (!$this->isRepeatable($annotationName)) {
            throw new LogicException(\sprintf(
                'The attribute "%s" is not repeatable. Call getPropertyAnnotation() instead.',
                $annotationName
            ));
        }

        return $this->getPropertyAnnotations($property)[$annotationName] ?? new RepeatableAttributeCollection();
    }

    /**
     * @param array<ReflectionAttribute> $attributes
     *
     * @template T of Annotation
     */
    private function convertToAttributeInstances(array $attributes): array
    {
        $instances = [];

        foreach ($attributes as $attribute) {
            $attributeName = $attribute->getName();
            \assert(\is_string($attributeName));
            // Make sure we only get Doctrine Annotations
            if (!\is_subclass_of($attributeName, Annotation::class)) {
                continue;
            }

            $instance = $attribute->newInstance();
            \assert($instance instanceof Annotation || $instance instanceof \Doctrine\ORM\Mapping\Annotation);

            if ($this->isRepeatable($attributeName)) {
                if (!isset($instances[$attributeName])) {
                    $instances[$attributeName] = new RepeatableAttributeCollection();
                }

                $collection = $instances[$attributeName];
                \assert($collection instanceof RepeatableAttributeCollection);
                $collection[] = $instance;
            } else {
                $instances[$attributeName] = $instance;
            }
        }

        return $instances;
    }

    /**
     * @param class-string<Annotation> $attributeClassName
     */
    private function isRepeatable(string $attributeClassName): bool
    {
        if (isset($this->isRepeatableAttribute[$attributeClassName])) {
            return $this->isRepeatableAttribute[$attributeClassName];
        }

        $reflectionClass = new ReflectionClass($attributeClassName);
        /** @var Attribute $attribute */
        $attribute = $reflectionClass->getAttributes()[0]->newInstance();

        return $this->isRepeatableAttribute[$attributeClassName] = ($attribute->flags & Attribute::IS_REPEATABLE) > 0;
    }
}
