<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Cyber\AuditBundle\Service\AuditManager;
use Doctrine\Common\Collections\Collection;
use Doctrine\Persistence\ObjectManager;

/**
 * @internal
 */
class AuditEventMapper
{
    /** @var ObjectManager */
    private $em;

    /** @var AuditManager<mixed> */
    private $auditManager;

    /**
     * AuditEventMapper constructor.
     *
     * @param ObjectManager       $em
     * @param AuditManager<mixed> $auditManager
     */
    public function __construct(ObjectManager $em, AuditManager $auditManager)
    {
        $this->em           = $em;
        $this->auditManager = $auditManager;
    }

    /**
     * @param AuditBuilder<mixed> $builder
     * @param object              $entity
     * @param int                 $depth
     */
    public function buildEventMap(AuditBuilder $builder, $entity, int $depth = 0): void
    {
        $metadata   = $this->em->getClassMetadata(\get_class($entity));
        $entityId   = $this->auditManager->getEntityIdClosure($entity, $metadata);
        $reflection = $metadata->getReflectionClass();

        $builder->mappedTo($reflection->getName(), $entityId);

        $auditMetadata = $this->auditManager->getAuditMetadata($metadata);

        if (null === $auditMetadata) {
            // if no audit stop recursion
            return;
        }

        $owners = $auditMetadata->getOwners();

        foreach ($owners as $owner) {
            // TODO on complex applications this can create a lot of extra select requests
            // should probably rework this to create a smart native query for getting a list of values for the map.

            if (0 === $depth) {
                $ownerChange = $builder->findOwnerChange($owner->getName());
                if (null !== $ownerChange) {
                    // only process owner changes for main entity which occurs at depth 0
                    $this->mapOwnerEntity($builder, $ownerChange[0], $depth); // map to old entity
                    $this->mapOwnerEntity($builder, $ownerChange[1], $depth); // map to new entity
                    continue;
                }
            }

            $owner->setAccessible(true);
            $ownerEntity = $owner->getValue($entity);

            $this->mapOwnerEntity($builder, $ownerEntity, $depth);
        }
    }

    /**
     * @param AuditBuilder<mixed> $builder
     * @param null|object         $ownerEntity
     * @param int                 $depth
     */
    private function mapOwnerEntity(AuditBuilder $builder, $ownerEntity, int $depth): void
    {
        if (null === $ownerEntity) {
            return;
        }

        if ($ownerEntity instanceof Collection) {
            // OneToMany/ManyToMany relationship marked as owner. Must map event to each of the entities
            foreach ($ownerEntity as $oEntity) {
                $this->buildEventMap($builder, $oEntity, $depth + 1);
            }

            return;
        }

        $this->buildEventMap($builder, $ownerEntity, $depth + 1);
    }
}
