<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\EventListener;

use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Cyber\AuditBundle\Service\AuditManager;
use Cyber\AuditBundle\Service\ChangeHydrator;
use Doctrine\Common\Annotations\Reader;
use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\ORM\Events;
use Doctrine\ORM\PersistentCollection;
use LogicException;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class AuditDoctrineSubscriber implements EventSubscriber
{
    /** @var Reader */
    private $reader;

    /** @var ChangeHydrator<mixed> */
    private $changeHydrator;

    /** @var AuditManager<mixed> */
    private $auditManager;

    /** @var null|AuditBuilder<mixed>[][] local cache of builders created during this flush operation */
    private $events;

    /**
     * @param ChangeHydrator<mixed> $changeHydrator
     * @param AuditManager<mixed>   $auditManager
     */
    public function __construct(Reader $reader, ChangeHydrator $changeHydrator, AuditManager $auditManager)
    {
        $this->reader         = $reader;
        $this->changeHydrator = $changeHydrator;
        $this->auditManager   = $auditManager;
    }

    /**
     * @inheritdoc
     */
    public function getSubscribedEvents(): array
    {
        return [
            Events::onFlush,
        ];
    }

    public function onFlush(OnFlushEventArgs $eventArgs): void
    {
        $em           = $eventArgs->getEntityManager();
        $uow          = $em->getUnitOfWork();
        $this->events = [];  // reset events for each flush operation

        foreach ($uow->getScheduledEntityInsertions() as $entity) {
            if ($entity instanceof Event || $entity instanceof Change || $entity instanceof EventMap) {
                // we do not want to process our audit entities as that will lead to infinite loops.
                continue;
            }
            $this->processEntity(Event::TYPE_INSERT, $entity, $eventArgs);
        }

        foreach ($uow->getScheduledEntityUpdates() as $entity) {
            $this->processEntity(Event::TYPE_UPDATE, $entity, $eventArgs);
        }

        foreach ($uow->getScheduledEntityDeletions() as $entity) {
            $this->processEntity(Event::TYPE_DELETE, $entity, $eventArgs);
        }

        foreach ($uow->getScheduledCollectionUpdates() as $collection) {
            if (!$collection instanceof PersistentCollection) {
                throw new LogicException('Expecting UOW to have persistent collections.');
            }
            $this->processCollection($collection, $eventArgs);
        }

        // we need to map events after change hydration but before flush completes
        // * change hydration populates owner changes
        // * after flush deleted entities will have *null* ids so audit will fail.
        // so we do it right before flush exists
        $this->finalizeBuilders($em);

        $this->events = null; // free up no longer needed cache
    }

    /**
     * @param int              $eventType
     * @param mixed            $entity
     * @param OnFlushEventArgs $eventArgs
     */
    private function processEntity($eventType, $entity, OnFlushEventArgs $eventArgs): void
    {
        $em  = $eventArgs->getEntityManager();
        $uow = $em->getUnitOfWork();

        $metadata = $em->getClassMetadata(\get_class($entity));

        $auditMetadata = $this->auditManager->getAuditMetadata($metadata);
        if (null === $auditMetadata) {
            return;
        }

        $builder = $this->getBuilder($entity, $eventType);

        switch ($eventType) {
            case Event::TYPE_INSERT:
            case Event::TYPE_UPDATE:
                $this->changeHydrator->fromChangeSet($builder, $auditMetadata, $uow->getEntityChangeSet($entity));
                break;
            case Event::TYPE_DELETE:
                $this->changeHydrator->fromOriginal($builder, $auditMetadata, $uow->getOriginalEntityData($entity));
        }
    }

    /**
     * @param PersistentCollection<array-key, object> $collection
     * @param OnFlushEventArgs                        $eventArgs
     */
    private function processCollection($collection, OnFlushEventArgs $eventArgs): void
    {
        $em = $eventArgs->getEntityManager();

        $owner = $collection->getOwner();
        if (null === $owner) {
            return;
        }

        $mapping = $collection->getMapping();
        if (null === $mapping) {
            return;
        }

        $metadata = $em->getClassMetadata(\get_class($owner));

        $auditMetadata = $this->auditManager->getAuditMetadata($metadata);
        if (null === $auditMetadata) {
            return;
        }

        $trackedProperties = $auditMetadata->getProperties();

        if (!$trackedProperties->contains($mapping['fieldName'])) {
            return;
        }

        if (!isset($mapping['joinTable'])) {
            // If there is no join table it is expected that the owning side will create the relevant audit mapping
            // related to the inverse side.
            return;
        }

        // for collections we want to get builder for owner of any event type
        $builder = $this->getBuilder($owner);

        $oldValues = $collection->getSnapshot();
        $newValues = $collection->toArray();

        if ([] === $oldValues && [] === $newValues) {
            // called clear() on collection
            $oldValues = ['*']; // TODO maybe we should somehow retrieve those values for proper audit and reverts
            $newValues = ['[deleted all]'];
        }

        $this->changeHydrator->fromChangeSet(
            $builder,
            $auditMetadata,
            [(string) $mapping['fieldName'] => [$oldValues, $newValues]]
        );
    }

    /**
     * @param object   $entity
     * @param null|int $eventType if null will try to find any cached builder for entity, otherwise only specific type
     *
     * @return AuditBuilder<mixed>
     */
    private function getBuilder($entity, $eventType = null): AuditBuilder
    {
        $hash = \spl_object_hash($entity);

        $cache = $this->events[$hash] ?? [];

        if (null !== $eventType && isset($cache[$eventType])) {
            return $cache[$eventType];
        }

        if (null === $eventType && !empty($cache)) {
            return \current($cache);
        }

        // if was null it means collection change of owner that itself did not change so treat as update in respect
        // to owner
        $eventType = null === $eventType ? Event::TYPE_UPDATE : $eventType;

        $builder = $this->auditManager->createEntityAuditBuilder($eventType, $entity);

        $this->auditManager->enqueueAudit($builder);

        return $this->events[$hash][$eventType] = $builder;
    }

    private function finalizeBuilders(EntityManagerInterface $em): void
    {
        if (null === $this->events) {
            // this should never happen
            throw new LogicException('The $events should not be null.');
        }

        // use locally cached events, we could potentially pull all queued audits, but the queue could have manually
        // created ones which would mess up the mapping most likely.
        foreach ($this->events as $builders) {
            foreach ($builders as $builder) {
                $this->auditManager->mapEvents($builder, $builder->getEntityObject(), $em);
            }
        }
    }
}
