<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\DependencyInjection;

use Cyber\AuditBundle\DependencyInjection\CyberAuditExtension;
use Cyber\AuditBundle\EventListener\AuditKernelListener;
use Cyber\AuditBundle\Service\EntityFactory;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * @internal
 */
class CyberAuditExtensionTest extends TestCase
{
    public function testEmptyLoad(): void
    {
        $extension = new CyberAuditExtension();
        $container = new ContainerBuilder();

        $extension->load([[]], $container);

        $this->assertFalse($container->hasDefinition(AuditKernelListener::class));
    }

    public function testMinimalConfigs(): void
    {
        $extension = new CyberAuditExtension();
        $container = new ContainerBuilder();

        $config = [
            'entity' => [
                'change' => 'C',
                'event'  => 'E',
                'map'    => 'M',
            ],
        ];

        $extension->load([$config], $container);

        $factory = $container->get('cyber.audit.entity_factory.service');
        static::assertInstanceOf(EntityFactory::class, $factory);
    }

    public function testCustomFactory(): void
    {
        $extension = new CyberAuditExtension();
        $container = new ContainerBuilder();

        $config = [
            'entity' => [
                'change'  => 'C',
                'event'   => 'E',
                'map'     => 'M',
                'factory' => DecoratedEntityFactory::class,
            ],
        ];

        $extension->load([$config], $container);

        $factory = $container->get('cyber.audit.entity_factory.service');
        static::assertInstanceOf(DecoratedEntityFactory::class, $factory);
        static::assertEquals(['C', 'E', 'M'], $factory->getParams());

        $extension = new CyberAuditExtension();
        $container = new ContainerBuilder();
        $container->register(DecoratedEntityFactory::class, DecoratedEntityFactory::class)
            ->setArguments(['X', 'Y', 'Z']);

        $config = [
            'entity' => [
                'change'  => 'C',
                'event'   => 'E',
                'map'     => 'M',
                'factory' => '@' . DecoratedEntityFactory::class,
            ],
        ];

        $extension->load([$config], $container);

        $factory = $container->get('cyber.audit.entity_factory.service');
        static::assertInstanceOf(DecoratedEntityFactory::class, $factory);
        static::assertEquals(['X', 'Y', 'Z'], $factory->getParams());
    }
}
