<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\Repository;

use Cyber\AuditBundle\Entity\Criteria\EntityCriteria;
use Cyber\AuditBundle\Entity\Criteria\EventCriteria;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Repository\EventRepository;
use DateTime;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Query\Expr;
use Doctrine\ORM\QueryBuilder;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * @internal
 *
 * @coversNothing
 */
class EventRepositoryTest extends TestCase
{
    /** @var EventRepository */
    private $repository;

    /** @var EntityManager|MockObject */
    private $em;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        // @var EntityManager|MockObject $em
        $this->em = $this->getMockBuilder(EntityManager::class)
            ->disableOriginalConstructor()
            ->getMock();

        /** @var ClassMetadata<Event<mixed, mixed>>|MockObject $meta */
        $meta       = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();
        $meta->name = Event::class;

        $this->repository = new EventRepository($this->em, $meta);
    }

    public function testMatch(): void
    {
        $queryBuilder = new QueryBuilder($this->em);
        $this->em->method('createQueryBuilder')
            ->willReturn($queryBuilder);
        $this->em->method('getExpressionBuilder')
            ->willReturn(new Expr());

        $event1 = new EntityCriteria();
        $event2 = new EntityCriteria();

        $event1->entityClass = Event::class;
        $event1->entityId    = 1;

        $event2->entityClass = Event::class;
        $event2->entityId    = 2;

        $criteria           = new EventCriteria();
        $criteria->actions  = [Event::TYPE_INSERT];
        $criteria->dateFrom = new DateTime('-10 days');
        $criteria->dateTo   = new DateTime('+10 days');
        $criteria->entities = [
            $event1,
            $event2,
        ];

        $queryBuilder = $this->repository->match($criteria);

        $dql = 'SELECT evt FROM Cyber\\AuditBundle\\Entity\\Event evt WHERE evt.type in (:eventTypes) AND evt.dateCreated >= :dateFrom AND ' .
            'evt.dateCreated < :dateTo';

        static::assertEquals($dql, $queryBuilder->getDQL());
    }

    /**
     * Remove in v4.
     *
     * @group legacy
     */
    public function testMatchDeprecated(): void
    {
        $queryBuilder = new QueryBuilder($this->em);
        $this->em->method('createQueryBuilder')
            ->willReturn($queryBuilder);
        $this->em->method('getExpressionBuilder')
            ->willReturn(new Expr());

        $event1 = new EntityCriteria();
        $event2 = new EntityCriteria();

        $event1->entityClass = Event::class;
        $event1->entityId    = 1;

        $event2->entityClass = Event::class;
        $event2->entityId    = 2;

        $criteria            = new EventCriteria();
        $criteria->actions   = [Event::TYPE_INSERT];
        $criteria->dateFrom  = new DateTime('-10 days');
        $criteria->dateTo    = new DateTime('+10 days');
        $criteria->entities  = [
            $event1,
            $event2,
        ];

        $queryBuilder = $this->repository->match($criteria);

        $dql = 'SELECT evt FROM Cyber\\AuditBundle\\Entity\\Event evt WHERE evt.type in (:eventTypes) AND evt.dateCreated >= :dateFrom AND ' .
            'evt.dateCreated < :dateTo';

        static::assertEquals($dql, $queryBuilder->getDQL());
    }

    public function testEmptyCriteria(): void
    {
        $queryBuilder = new QueryBuilder($this->em);
        $this->em->method('createQueryBuilder')
            ->willReturn($queryBuilder);
        $this->em->method('getExpressionBuilder')
            ->willReturn(new Expr());

        $criteria     = new EventCriteria();
        $queryBuilder = $this->repository->match($criteria);

        $dql = 'SELECT evt FROM Cyber\\AuditBundle\\Entity\\Event evt';

        static::assertEquals($dql, $queryBuilder->getDQL());
    }
}
