<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Closure;
use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\ChangeValue;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Service\EntityFactory;
use Doctrine\Common\Collections\Collection;
use Generator;
use LogicException;

/**
 * @template T
 * @template IdType
 *
 * @phpstan-type CallableId callable(): IdType
 * @phpstan-type CallableValue callable(): scalar
 *
 * @SuppressWarnings(PHPMD.TooManyPublicMethods)
 */
class AuditBuilder
{
    /** @var EntityFactory<T, IdType> */
    private $factory;

    /** @var int */
    private $eventType;

    /** @var null|string */
    private $entityClass;

    /** @var string */
    private $alias;

    /** @var null|CallableId|IdType|non-empty-array<CallableId|IdType> */
    private $id;

    /** @var null|string */
    private $description;

    /** @var null|string */
    private $message;

    /** @var Change[] */
    private $changes = [];

    /** @var array<class-string, non-empty-array<callable(): IdType|IdType>> */
    private $maps = [];

    /** @var array<string, array<int, mixed>|Collection<int, mixed>> */
    private $ownerChanges = [];

    /** @var null|object */
    private $entityObject;

    /**
     * @param EntityFactory<T, IdType> $factory
     * @param null|object              $entityObject
     */
    public function __construct(EntityFactory $factory, $entityObject = null)
    {
        $this->factory      = $factory;
        $this->entityObject = $entityObject;
    }

    /**
     * @return $this
     */
    public function created(string $class): self
    {
        return $this->eventFor(Event::TYPE_INSERT, $class);
    }

    /**
     * @return $this
     */
    public function updated(string $class): self
    {
        return $this->eventFor(Event::TYPE_UPDATE, $class);
    }

    /**
     * @return $this
     */
    public function deleted(string $class): self
    {
        return $this->eventFor(Event::TYPE_DELETE, $class);
    }

    /**
     * @return $this
     */
    public function softDeleted(string $class): self
    {
        return $this->eventFor(Event::TYPE_SOFT_DELETE, $class);
    }

    /**
     * @return $this
     */
    public function messageFor(string $class): self
    {
        return $this->eventFor(Event::TYPE_MESSAGE, $class);
    }

    /**
     * @return $this
     */
    public function eventFor(int $type, string $class): self
    {
        $this->entityClass = $class;
        $this->eventType   = $type;

        return $this;
    }

    /**
     * Let's you override the event type for this audit.
     *
     * For example it may have started as UPDATE, but what updated was soft delete field
     * so you would need to change the type to SOFT_DELETED
     *
     * @param int $type
     *
     * @return $this
     */
    public function ofType(int $type): self
    {
        $this->eventType = $type;

        return $this;
    }

    /**
     * @param null|CallableId|IdType|non-empty-array<CallableId|IdType> $id
     *
     * @return $this
     */
    public function withId($id): self
    {
        $this->id = $id;

        return $this;
    }

    /**
     * @return $this
     */
    public function alias(string $alias): self
    {
        $this->alias = $alias;

        return $this;
    }

    /**
     * @return $this
     */
    public function description(?string $description): self
    {
        $this->description = $description;

        return $this;
    }

    /**
     * @return $this
     */
    public function message(string $message): self
    {
        $this->message = $message;

        return $this;
    }

    /**
     * @param string                                   $field
     * @param null|callable|callable[]|string|string[] $newValue
     * @param null|callable|callable[]|string|string[] $oldValue
     * @param null|callable|callable[]|string|string[] $newUserValue
     * @param null|callable|callable[]|string|string[] $oldUserValue
     *
     * @return $this
     */
    public function changed(
        string $field,
        $newValue,
        $oldValue = null,
        $newUserValue = null,
        $oldUserValue = null
    ): self {
        $oldChangeVal = new ChangeValue($oldValue, $oldUserValue);
        $newChangeVal = new ChangeValue($newValue, $newUserValue);

        $change = $this->factory->createChange();
        $change->setField($field)
            ->setNewValue($newChangeVal)
            ->setOldValue($oldChangeVal);

        $this->changes[] = $change;

        return $this;
    }

    /**
     * @param class-string              $class
     * @param callable(): IdType|IdType $id
     *
     * @return $this
     */
    public function mappedTo(string $class, $id): self
    {
        if(!isset($this->maps[$class])) {
            $this->maps[$class] = [$id];

            return $this;
        }
        $this->maps[$class][] = $id;

        return $this;
    }

    /**
     * @param Event<T, IdType> $event
     *
     * @return Generator<\Cyber\AuditBundle\Entity\EventMap<T, IdType>>
     */
    private function makeMaps(Event $event): Generator
    {
        foreach($this->maps as $class => $ids) {
            foreach($ids as $id) {
                yield $this->factory->createMap($event)
                    ->setEntityClass($class)
                    ->setEntityId($this->resolveIdValue($id));
            }
        }
    }

    /**
     * @return Event<T, IdType>
     */
    public function getEvent(): Event
    {
        $event = $this->factory->createEvent();

        $event->setType($this->eventType)
            ->setEntityClass($this->getEntityClass())
            ->setEntityAlias($this->alias)
            ->setEntityId($this->resolveIdValue($this->id))
            ->setDescription($this->description)
            ->setMessage($this->message);

        foreach ($this->changes as $change) {
            $oldValue = $change->getOldValue();
            $newValue = $change->getNewValue();

            $oldValue->setRawValue($this->resolveValue($oldValue->getRawValue()));
            $oldValue->setUserValue($this->resolveValue($oldValue->getUserValue()));
            $newValue->setRawValue($this->resolveValue($newValue->getRawValue()));
            $newValue->setUserValue($this->resolveValue($newValue->getUserValue()));

            $event->addChange($change);
        }

        foreach ($this->makeMaps($event) as $map) {
            $event->addMap($map);
        }

        return $event;
    }

    public function isValid(): bool
    {
        switch ($this->eventType) {
            case Event::TYPE_MESSAGE:
                // if message with empty message it is not valid
                return !empty($this->message);
            case Event::TYPE_SOFT_DELETE:
                // no special requirements for soft deletion
                return true;
            default:
                // if change with no changes it is invalid
                return isset($this->changes[0]);
        }
    }

    public function getEntityClass(): string
    {
        return $this->entityClass ?? '';
    }

    /**
     * @return $this
     */
    public function validate(): self
    {
        switch (true) {
            case null === $this->eventType:
                throw $this->invalidException(
                    'event type',
                    ['created()', 'updated()', 'deleted()', 'softDeleted()', 'messageFor()', 'eventFor()']
                );
            case null === $this->entityClass:
                throw $this->invalidException(
                    'entity class',
                    ['created()', 'updated()', 'deleted()', 'softDeleted()', 'messageFor()', 'eventFor()']
                );
            case null === $this->alias:
                throw $this->invalidException('alias', ['alias()']);
            case null === $this->id:
                throw $this->invalidException('id', ['withId()']);
        }

        return $this;
    }

    public function validateMapping(): void
    {
        if (empty($this->maps)) {
            throw $this->invalidException('maps', ['mappedTo() at least once (for self mapping)']);
        }
    }

    /**
     * @param string                                   $field
     * @param array<int, mixed>|Collection<int, mixed> $valueChange
     */
    public function ownerChange(string $field, $valueChange): void
    {
        $this->ownerChanges[$field] = $valueChange;
    }

    /**
     * @param string $field
     *
     * @return null|array<int, mixed>|Collection<int, mixed>
     */
    public function findOwnerChange(string $field)
    {
        return $this->ownerChanges[$field] ?? null;
    }

    /**
     * @return null|object
     */
    public function getEntityObject()
    {
        return $this->entityObject;
    }

    /**
     * @param string   $key
     * @param string[] $methods
     *
     * @return LogicException
     */
    private function invalidException(string $key, array $methods): LogicException
    {
        $message = \sprintf('Missing %s, you must call one of: %s', $key, \implode(', ', $methods));

        return new LogicException($message);
    }

    /**
     * If value is callable returns the result of the call, otherwise returns value.
     *
     * @param null|CallableValue|CallableValue[]|scalar|scalar[] $value
     */
    private function resolveValue($value): ?string
    {
        if(null === $value) {
            return null;
        }

        if ($value instanceof Closure) {
            $value = $value();
        }

        if (\is_array($value)) {
            return \json_encode(\array_map([$this, 'resolveValue'], $value)) ?: 'json_encode_failed';
        }

        return (string) $value;
    }

    /**
     * If value is callable returns the result of the call, otherwise returns value.
     *
     * @param null|CallableId|IdType|non-empty-array<CallableId|IdType> $value
     *
     * @return non-empty-array<IdType>
     */
    private function resolveIdValue($value)
    {
        if ($value instanceof Closure) {
            $value = $value();
        }

        if(!\is_array($value)) {
            return [$value];
        }

        $resolved = [];

        foreach($value as $item) {
            // resulting array will always contain 1 item.
            $resolved[] = $this->resolveIdValue($item)[0];
        }

        // since value is a non-empty-array, resolved will always have at least 1 item
        // for some reason Stan is not realizing that, so just add this assert here.
        \assert(!empty($resolved), 'Resolved ID is somehow empty, this should not be possible');

        return $resolved;
    }
}
