# About
This bundle provides easy to use auditing capabilities. 

# Flex
If you are using symfony flex, default configuration and entity files will be automatically created for you.

# PhpStorm Annotation Alias
We recommend adding an annotation alias to your PhpStorm configuration so it auto-aliases them when using inside your
entities. In setting *Languages & Frameworks > PHP > Annotations > Use Alias* add the following:

| Class | Alias |
|-------|-------|
| Cyber\AuditBundle\Annotation | Audit |

# Configuration

```yaml
cyber_audit:
    entity:               # Required

        # Class name of the Change entity
        change: App\Entity\AuditChange # Required

        # Class name of the Event entity
        event: App\Entity\AuditEvent   # Required

        # Class name of the EventMap entity
        map: App\Entity\AuditMap       # Required
```

Additionally this bundle depends on user supplier from OrmExtras bundle. Its configuration can be found in
[bundles documentation](https://gitlab.cybercoder.site/ghcm/cyber/OrmExtrasBundle/blob/master/src/Resources/doc/user-supplier.md)

# Entities
First, you need to create entities to store audit changes, events, and maps. These entities need to extend the base
entities from this bundle: 
* Cyber\AuditBundle\Entity\Change
* Cyber\AuditBundle\Entity\Event
* Cyber\AuditBundle\Entity\EventMap

> Flex users will have those entities auto-generated in `App\Entity\{AuditChange|AuditEvent|AuditMap}` 

You need to provide column configuration all reference columns in these entities. You may override any of
the other columns if you are not satisfied with their default names.

> If you choose not to use `UuidEntityTrait` (from cyber/orm-extras-bundle), then you also need to provide field and
> column definition for the for primary key column of each of the above entities. 

## Composite Keys
If your DB model contains composite primary keys, you will need to override `Event->setEntityId()` and 
`EventMap->setEntityId()` as its default behavior is set in the way to support single column primary keys. 
You will also need to configure your id column accordingly.

## Minimal Configs

### With UuidEntityTrait
```php
<?php

namespace App\Entity;

use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Cyber\OrmExtras\Utility\UuidEntityTrait;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity
 * @ORM\Table(name="audit_change")
 */
class AuditChange extends Change
{
    use UuidEntityTrait;

    /**
     * @var AuditEvent
     * @ORM\ManyToOne(targetEntity="App\Entity\AuditEvent", inversedBy="changes")
     * @ORM\JoinColumn(name="ach_atr_id", referencedColumnName="aev_id", nullable=false)
     */
    protected $event;
}

/**
 * @ORM\Entity
 * @ORM\Table(name="audit_event")
 */
class AuditEvent extends Event
{
    use UuidEntityTrait;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\User")
     * @ORM\JoinColumn(name="aev_usr_id", referencedColumnName="usr_id", nullable=false)
     * @Serializer\Groups({"Default", "audit-simple", "audit-full"})
     */
    protected $userCreated;

    /**
     * @var AuditChange
     * @ORM\OneToMany(targetEntity="App\Entity\AuditChange", mappedBy="event", cascade={"persist"})
     * @Serializer\Groups({"Default", "audit-simple", "audit-full"})
     */
    protected $changes;

    /**
     * @var AuditMap
     * @ORM\OneToMany(targetEntity="App\Entity\AuditMap", mappedBy="event", cascade={"persist"})
     * @Serializer\Exclude
     */
    protected $maps;
}

/**
 * @ORM\Entity
 * @ORM\Table(name="audit_map")
 */
class AuditMap extends EventMap
{
    use UuidEntityTrait;

    /**
     * @var AuditEvent
     * @ORM\ManyToOne(targetEntity="App\Entity\AuditEvent", inversedBy="maps")
     * @ORM\JoinColumn(name="aem_aev_id", referencedColumnName="aev_id")
     */
    protected $event;
}
```

### WithOUT UuidEntityTrait

```php
<?php

namespace App\Entity;

use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity
 * @ORM\Table(name="audit_change")
 */
class AuditChange extends Change
{
    /**
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     * @ORM\Column(name="ach_id", type="integer")
     */
    protected $id;

    /**
     * @var AuditEvent
     * @ORM\ManyToOne(targetEntity="App\Entity\AuditEvent", inversedBy="changes")
     * @ORM\JoinColumn(name="ach_atr_id", referencedColumnName="aev_id", nullable=false)
     */
    protected $event;

    public function getId()
    {
        return $this->id;    
    }
}

/**
 * @ORM\Entity
 * @ORM\Table(name="audit_event")
 */
class AuditEvent extends Event
{
    /**
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     * @ORM\Column(name="aev_id", type="integer")
     * @Serializer\Groups({"audit-full"})
     */
    protected $id;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\User")
     * @ORM\JoinColumn(name="aev_usr_id", referencedColumnName="usr_id", nullable=false)
     * @Serializer\Groups({"Default", "audit-simple", "audit-full"})
     */
    protected $userCreated;

    /**
     * @var AuditChange
     * @ORM\OneToMany(targetEntity="App\Entity\AuditChange", mappedBy="event", cascade={"persist"})
     * @Serializer\Groups({"Default", "audit-simple", "audit-full"})
     */
    protected $changes;

    /**
     * @var AuditMap
     * @ORM\OneToMany(targetEntity="App\Entity\AuditMap", mappedBy="event", cascade={"persist"})
     * @Serializer\Exclude
     */
    protected $maps;

    public function getId()
    {
        return $this->id;    
    }
}

/**
 * @ORM\Entity
 * @ORM\Table(name="audit_map")
 */
class AuditMap extends EventMap
{
    /**
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     * @ORM\Column(name="aem_id", type="integer")
     */
    protected $id;

    /**
     * @var AuditEvent
     * @ORM\ManyToOne(targetEntity="App\Entity\AuditEvent", inversedBy="maps")
     * @ORM\JoinColumn(name="aem_aev_id", referencedColumnName="aev_id")
     */
    protected $event;

    public function getId()
    {
        return $this->id;    
    }
}
```

> Note that *userCreated* should reference the entity that is returned by the UserSupplier mentioned in the Configuration 
> section. If your UserSupplier returns string username, then this should be a simple *string* column.

## EntityFactory

If you need more customization over audit entity creation, you can decorate the EntityFactory class and provide your
implementation via configs

```yaml
cyber_audit:
    entity:
        factory: "class_name OR @service"
```

# Auditing Changes
You are now ready to start using the bundle to audit changes. This bundle uses opt-in functionality, so by default
nothing gets audited. In order to audit a change you need to apply appropriate attributes to the entity to be audited.

## @Audit\Apply Attribute
This is the main attribute that has to be set on the entity class. It indicates to the audit system that it should look
at this entity for potential audit needs.

This attribute has 2 optional properties that affect the behavior of the audit:

| Property | Description |
| -- | -- |
| displayName | A user friendly name of the entity. If not provided, entity's short class name will be used. |
| inclusionCriteria | Has 2 possible values: `ALL` (default) and `NONE` |

If you specify `ALL` for **inclusionCriteria** (or omit it), all properties will be marked for audit, and you can apply
the `@Audit\Skip` attribute for any property you wish to exclude.

If you specify `NONE` for **inclusionCriteria**, none of the properties will be marked for audit, and you will need to
apply the `@Audit\Track` attribute to include individual properties into the audit.

## @Audit\Skip Attribute
If applied to a property will exclude it from being audited.

## @Audit\Track Attribute
If applied to a property will include it in the audit log.

## @Audit\Cascade Attribute
This attribute can be applied to properties that have OneToOne, OneToMany, ManyToOne, and ManyToMany relationships.

If applied to a property it will cause any changes that were audited for this entity to also be mapped as changes to 
the entity(ies) referenced by this property.

## Examples

Let's say we have the following entity configuration:

```php
<?php

use Cyber\AuditBundle\Annotation as Audit;

/**
 * @ORM\Entity
 *
 * @Audit\Apply(inclusionCriteria="NONE")
 */
class Patient
{

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=255, unique=true)
     * @Audit\Track
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=45, nullable=true)
     */
    private $street;
    
    /**
     * @var ArrayCollection | Shipment[]
     *
     * @ORM\OneToMany(targetEntity="App\Entity\Shipment", mappedBy="patient")
     */
    private $shipments;
}

/**
 * @ORM\Entity
 * @Audit\Apply
 */
class Shipment
{

    /**
     * @var int
     *
     * @ORM\Column(name="cpr_id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="cpr_unit_cogs", type="decimal", precision=8, scale=2, nullable=true)
     */
    private $cost;

    /**
     * @var Patient
     *
     * @ORM\ManyToOne(targetEntity="App\Entity\Patient", inversedBy="shipments")
     * @Audit\Cascade
     */
    private $patient;

}
```

With the above configuration:
* *name* changes to `Patient` record will be audited under `Patient`
* all property changes to `Shipment` will be audited under `Shipment` and the `Patient` that is set in the **$patient**
  property at the time of audit.

# Fetching audit changes
The `Cyber\AuditBundle\Service\AuditManager` is used internally to create audits but also provides a way for you to 
fetch audit changes for most common situations.  

`AuditManager::findEntityAuditsQb($entityCriteria, $forUser = null)` takes one or an array of 
`Cyber\AuditBundle\Entity\Criteria\EventCriteria` as the first argument and optional user value as second argument.

The `EventCriteria` lets you specify single entity and optionally an id for that entity to return the audit events for.
If you specify multiple `EventCriteria` results for multiple entities with multiple ids will get returned.
If *$forUser* is set, only events related to this user are returned.

## Example

```php
<?php

use Cyber\AuditBundle\Entity\Criteria\EventCriteria;
use Cyber\AuditBundle\Service\AuditManager;

$user = $fetchUserFromDb;

$criteria1 = new EventCriteria();

$criteria1->entityClass = Patient::class;
$criteria1->entityId    = '11589';

$criteria2 = new EventCriteria();
          
$criteria1->entityClass = Shipment::class;

/** @var AuditManager $manager */
$events = $manager->findEntityAuditsQb([$criteria1, $criteria2], $user)->getQuery()->execute();
/** @var AuditEvent[] $events */
```

The **$events** will contain an array of change events caused by the **$user** related to patient *11589* and 
absolutely all shipments in the system. 

> The above lookup will probably not be useful, but it is there just to demonstrate the way to fetch events related to
> all entities of particular type, not just a single id. 


# Disable audit for individual entities
The `Cyber\AuditBundle\Service\AuditManager` allows you to disable individual entities using the following functions:
`AuditManager::isDisableForEntity(string $entityClass)` takes a string containing the name of the class to be disabled and checks for disable audit for entity;
`AuditManager::disableForEntity(string $entityClass)` takes a string containing the name of the class to be disabled and disable audit for entity;
`AuditManager::enableForEntity(string $entityClass)` takes a string containing the name of the class to be disabled and enable audit for entity;.


## Example

```php
<?php

use Cyber\AuditBundle\Service\AuditManager;
use Entity;

// Example with add entity:
$patient = new Entity\Patient(); 
$comment  = new Entity\Comment();

/** @var AuditManager $manager */ 
$manager->disableForEntity($comment); // if not need audit comment for patient.

$comment->setText('test comment');
$patient->addComment($comment);

$em->persist($patient); 
$em->flush(); //Entity\Comment won't audited 

// Example with edit entity:
$patient = $em->find(Entity\Patient::class, 1);

$patient->removeComment($comment);

if ($manager->isDisableForEntity($comment)){
    $manager->enableForEntity($comment); 
}

$em->flush(); //Entity\Comment will audited 
```
