<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Entity;

use Cyber\OrmExtras\Utility\CreatorAttributes;
use Cyber\OrmExtras\Utility\CreatorAttributeTrait;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

/**
 * @template T
 * @template IdType
 *
 * @phpstan-type CallableId callable(): IdType
 *
 * @implements CreatorAttributes<T>
 */
#[ORM\MappedSuperclass]
class Event implements CreatorAttributes
{
    /** @phpstan-use CreatorAttributeTrait<T> */
    use CreatorAttributeTrait;

    public const TYPE_INSERT = 0;

    public const TYPE_UPDATE = 25;

    public const TYPE_DELETE = 50;

    public const TYPE_SOFT_DELETE = 75;

    public const TYPE_MESSAGE = 100;

    public const TYPE_NAMES = [
        self::TYPE_INSERT      => 'Created',
        self::TYPE_UPDATE      => 'Updated',
        self::TYPE_DELETE      => 'Deleted',
        self::TYPE_SOFT_DELETE => 'Soft Deleted',
        self::TYPE_MESSAGE     => 'Message',
    ];

    protected ?DateTimeInterface $dateCreated = null;

    /**
     * @var ArrayCollection<int, Change>|Change[]
     */
    protected Collection $changes;

    /**
     * @var ArrayCollection<int, EventMap<T, IdType>>
     */
    protected Collection $maps;

    #[ORM\Column(type: Types::STRING, length: 255, nullable: false)]
    protected string $entityClass = '';

    /**
     * @var null|IdType|IdType[]
     */
    protected $entityId;

    /**
     * A user friendly entity name.
     */
    #[ORM\Column(type: Types::STRING, length: 80, nullable: false)]
    protected string $entityAlias = '';

    /**
     * A user friendly target name.
     */
    #[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
    protected ?string $description = null;

    #[ORM\Column(type: Types::INTEGER, nullable: false)]
    protected int $type = 0;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    protected ?string $message = null;

    public function __construct()
    {
        $this->maps    = new ArrayCollection();
        $this->changes = new ArrayCollection();
    }

    /**
     * @return string
     */
    public function getEntityClass(): ?string
    {
        return $this->entityClass;
    }

    /**
     * @param string $entityClass
     *
     * @return $this
     */
    public function setEntityClass(string $entityClass): self
    {
        $this->entityClass = $entityClass;

        return $this;
    }

    /**
     * @return null|CallableId|CallableId[]|IdType|IdType[]
     */
    public function getEntityId()
    {
        return $this->entityId;
    }

    /**
     * @param non-empty-array<IdType> $entityId
     *
     * @return $this
     */
    public function setEntityId(array $entityId): self
    {
        $this->entityId = $entityId[0];

        return $this;
    }

    /**
     * @return string
     */
    public function getEntityAlias(): ?string
    {
        return $this->entityAlias;
    }

    /**
     * @param string $entityAlias
     *
     * @return $this
     */
    public function setEntityAlias(string $entityAlias): self
    {
        $this->entityAlias = $entityAlias;

        return $this;
    }

    /**
     * @return int
     */
    public function getType(): ?int
    {
        return $this->type;
    }

    /**
     * Type of event that occurred (ex. Created).
     *
     * @return string
     */
    public function getTypeLabel(): ?string
    {
        return self::TYPE_NAMES[$this->type] ?? 'None';
    }

    /**
     * @param int $type
     *
     * @return $this
     */
    public function setType(int $type): self
    {
        $this->type = $type;

        return $this;
    }

    /**
     * @return ArrayCollection<int,Change>|Change[]
     */
    public function getChanges()
    {
        return $this->changes;
    }

    /**
     * @param Change $change
     *
     * @return $this
     */
    public function addChange(Change $change): self
    {
        $change->setEvent($this);
        $this->changes[] = $change;

        return $this;
    }

    public function removeChange(Change $change): void
    {
        $this->changes->removeElement($change);
    }

    /**
     * @return string
     */
    public function getMessage(): ?string
    {
        return $this->message;
    }

    /**
     * @param string $message
     *
     * @return $this
     */
    public function setMessage(?string $message): self
    {
        $this->message = $message;

        return $this;
    }

    /**
     * @param EventMap<T, IdType> $map
     *
     * @return $this
     */
    public function addMap(EventMap $map): self
    {
        $this->maps[] = $map;

        return $this;
    }

    /**
     * @param EventMap<T, IdType> $map
     */
    public function removeMap(EventMap $map): void
    {
        $this->maps->removeElement($map);
    }

    /**
     * @return ArrayCollection<int,EventMap<T, IdType>>|EventMap<T, IdType>[]
     */
    public function getMaps()
    {
        return $this->maps;
    }

    /**
     * @return null|string
     */
    public function getDescription(): ?string
    {
        return $this->description;
    }

    /**
     * @param string $description
     *
     * @return $this
     */
    public function setDescription(?string $description): self
    {
        $this->description = $description;

        return $this;
    }
}
