<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Service;

use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;

/**
 * @template T
 * @template IdType
 *
 * @internal
 */
class EntityFactory
{
    /** @var class-string<Change> */
    protected string $changeClass;

    /** @var class-string<Event<T, IdType>> */
    protected string $eventClass;

    /** @var class-string<EventMap<T, IdType>> */
    protected string $mapClass;

    /**
     * EntityFactory constructor.
     *
     * @param class-string<Change>              $changeClass
     * @param class-string<Event<T, IdType>>    $eventClass
     * @param class-string<EventMap<T, IdType>> $mapClass
     */
    public function __construct(string $changeClass, string $eventClass, string $mapClass)
    {
        $this->changeClass = $changeClass;
        $this->eventClass  = $eventClass;
        $this->mapClass    = $mapClass;
    }

    /**
     * @return Event<T, IdType>
     */
    public function createEvent(): Event
    {
        return new $this->eventClass();
    }

    /**
     * @return Change
     */
    public function createChange(): Change
    {
        return new $this->changeClass();
    }

    /**
     * @param Event<T, IdType> $event
     *
     * @return EventMap<T, IdType>
     */
    public function createMap(Event $event): EventMap
    {
        return new $this->mapClass($event);
    }

    /**
     * @return class-string<Change>
     */
    public function getChangeClass()
    {
        return $this->changeClass;
    }

    /**
     * @return class-string<Event<T, IdType>>
     */
    public function getEventClass()
    {
        return $this->eventClass;
    }

    /**
     * @return class-string<EventMap<T, IdType>>
     */
    public function getMapClass()
    {
        return $this->mapClass;
    }
}
