<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\Service;

use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\Describer\ToStringDescriber;
use Cyber\AuditBundle\Entity\ChangeValue;
use Cyber\AuditBundle\Service\AuditManager;
use Cyber\AuditBundle\Service\ChangeValueTransformer;
use DateTime;
use DateTimeImmutable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\Persistence\ManagerRegistry;
use Exception;
use InvalidArgumentException;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Tests\Cyber\AuditBundle\Mock\MockComment;
use Tests\Cyber\AuditBundle\Mock\MockUser;
use Tests\Cyber\AuditBundle\Mock\MockUserTarget;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 *
 * @internal
 *
 * @coversNothing
 */
class ChangeValueTransformerTest extends TestCase
{
    /** @var ChangeValueTransformer<mixed, mixed> */
    private $transformer;

    /** @var ManagerRegistry|MockObject */
    private $registry;

    /** @var AuditManager<mixed, mixed>|MockObject */
    private $auditManager;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->registry     = $this->getMockBuilder(ManagerRegistry::class)->getMock();
        $this->auditManager = $this->getMockBuilder(AuditManager::class)->disableOriginalConstructor()->getMock();
        $this->transformer  = new ChangeValueTransformer($this->registry, $this->auditManager);
    }

    /**
     * @dataProvider scalarData
     *
     * @param string                                   $field
     * @param null|object|scalar                       $value
     * @param null|callable|callable[]|string|string[] $expected
     */
    public function testTransformScalar($field, $value, $expected): void
    {
        /** @var AuditMetadata<object>|MockObject $auditMeta */
        $auditMeta = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();

        $result      = $this->transformer->transformField($field, $value, $auditMeta);
        $expectedObj = new ChangeValue($expected, $expected);
        static::assertEquals($expectedObj, $result);
    }

    /**
     * @throws Exception
     *
     * @return array<array<mixed>>
     */
    public function scalarData(): array
    {
        return [
            ['string', 'simple string', 'simple string'],
            ['dateonly', new DateTime('2018-05-27 00:00:00'), '2018-05-27'],
            ['datetime', new DateTimeImmutable('2018-05-27 01:00:00'), '2018-05-27 01:00:00'],
            ['int', 5, '5'],
            ['float', 5.5, '5.5'],
            ['bool', false, 'false'],
            ['null', null, '(null)'],
        ];
    }

    public function testTransformScalarObject(): void
    {
        $this->expectException(InvalidArgumentException::class);

        $this->transformer->transformScalar((object) ['someField' => 'name']);
    }

    /**
     * @dataProvider entityProvider
     *
     * @param object $entity
     * @param mixed  $expectRaw
     * @param mixed  $expectUser
     */
    public function testTransformEntity($entity, $expectRaw, $expectUser): void
    {
        /** @var ClassMetadata<MockUser>|MockObject $metadata */
        $metadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();
        /** @var AuditMetadata<object>|MockObject $auditMeta */
        $auditMeta = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();

        $this->auditManager->expects(static::once())
            ->method('getEntityIdClosure')
            ->with($entity, $metadata)
            ->willReturnCallback(function (MockUser $entity) {
                return function () use ($entity) {
                    return $entity->getId();
                };
            });

        $this->auditManager->expects(static::any())
            ->method('getAuditMetadata')
            ->with($metadata)
            ->willReturn($auditMeta);
        $auditMeta->expects(static::any())
            ->method('describe')
            ->willReturnCallback(function ($value) {
                return (new ToStringDescriber())->describe($value);
            });

        $result = $this->transformer->transformEntity($entity, $metadata);

        $raw = $result->getRawValue();

        static::assertIsCallable($raw, '$raw not callable');

        static::assertEquals($expectRaw, $raw());
        static::assertEquals($expectUser, $result->getUserValue());
    }

    /**
     * @return array<array<mixed>>
     */
    public function entityProvider(): array
    {
        return [
            [
                (new MockUser())->setName('test')->setId(1),
                1,
                'test',
            ],
            [
                (new MockUserTarget())->setName('test')->setId(1),
                1,
                'test',
            ],
        ];
    }

    public function testTransformScalarField(): void
    {
        /** @var AuditMetadata<object>|MockObject $metadata */
        $metadata = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();
        $metadata->expects(static::any())
            ->method('getUserValues')
            ->withConsecutive(['id'], ['emails'])
            ->willReturnOnConsecutiveCalls([1 => 'root'], null);

        $result = $this->transformer->transformField('id', 1, $metadata);

        static::assertEquals(1, $result->getRawValue());
        static::assertEquals('root', $result->getUserValue());

        $result = $this->transformer->transformField('emails', ['a@a.com', 'b@b.com'], $metadata);

        static::assertEquals(['a@a.com', 'b@b.com'], $result->getRawValue());
        static::assertEquals(['a@a.com', 'b@b.com'], $result->getUserValue());
    }

    public function testTransformSimpleObjects(): void
    {
        /** @var AuditMetadata<object>|MockObject $metadata */
        $metadata = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();

        $comments = new ArrayCollection();
        $comments->add((new MockComment())->setId(1)->setComment('com1'));
        $comments->add((new MockComment())->setId(2)->setComment('com2'));

        $result = $this->transformer->transformField('comments', $comments, $metadata);

        $rawValues = $result->getRawValue();

        static::assertEquals(['com1', 'com2'], $rawValues);
        static::assertEquals(['com1', 'com2'], $result->getUserValue());
    }

    public function testTransformCollectionField(): void
    {
        /** @var AuditMetadata<object>|MockObject $metadata */
        $metadata = $this->getMockBuilder(AuditMetadata::class)->disableOriginalConstructor()->getMock();
        /** @var ClassMetadata<MockComment>|MockObject $classMetadata */
        $classMetadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();
        /** @var ClassMetadata<MockComment>|MockObject $childMetadata */
        $childMetadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();

        /** @var EntityManagerInterface|MockObject $manager */
        $manager = $this->getMockBuilder(EntityManagerInterface::class)->getMock();

        $metadata->expects(static::any())
            ->method('getClassMetadata')
            ->willReturn($classMetadata);

        $classMetadata->expects(static::any())
            ->method('hasAssociation')
            ->with('comments')
            ->willReturn(true);

        $classMetadata->expects(static::any())
            ->method('getAssociationTargetClass')
            ->with('comments')
            ->willReturn(MockComment::class);

        $this->registry->expects(static::any())
            ->method('getManagerForClass')
            ->with(MockComment::class)
            ->willReturn($manager);

        $manager->expects(static::any())
            ->method('getClassMetadata')
            ->with(MockComment::class)
            ->willReturn($childMetadata);

        $this->auditManager->expects(static::any())
            ->method('getEntityIdClosure')
            ->willReturnCallback(function (MockComment $comment) {
                return function () use ($comment) {
                    return $comment->getId();
                };
            });

        // technically this should be different metadata object as it is for different entity, but for purposes of this tests this will suffice
        $this->auditManager->expects(static::any())
            ->method('getAuditMetadata')
            ->with($childMetadata)
            ->willReturn($metadata);
        $metadata->expects(static::any())
            ->method('describe')
            ->willReturnCallback(function ($value) {
                return (new ToStringDescriber())->describe($value);
            });

        $comments = new ArrayCollection();
        $comments->add((new MockComment())->setId(1)->setComment('com1'));
        $comments->add((new MockComment())->setId(2)->setComment('com2'));

        $result = $this->transformer->transformField('comments', $comments, $metadata);

        $rawValues = (array) $result->getRawValue();

        static::assertCount(2, $rawValues);

        $resolved = \array_map(function ($callback) {
            return $callback();
        }, $rawValues);

        static::assertEquals(['1', '2'], $resolved);
        static::assertEquals(['com1', 'com2'], $result->getUserValue());
    }
}
