<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle\Service;

use Closure;
use Cyber\AuditBundle\Annotation\Apply;
use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\AuditEventMapper;
use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\AuditMetadataFactory;
use Cyber\AuditBundle\Entity\Criteria\EventCriteria;
use Cyber\AuditBundle\Repository\EventRepository;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\Query\Expr;
use Doctrine\Persistence\ManagerRegistry;
use Doctrine\Persistence\Mapping\ClassMetadata;
use Doctrine\Persistence\ObjectManager;
use Generator;
use InvalidArgumentException;
use SplDoublyLinkedList;
use Stringable;

/**
 * @template T
 * @template IdType
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class AuditManager
{
    /** @var EntityFactory<T, IdType> */
    private $entityFactory;

    /** @var ManagerRegistry */
    private $doctrine;

    /** @var AuditMetadataFactory */
    private $metadataFactory;

    /** @var AuditQueue<AuditBuilder> */
    private $queue;

    /** @var array<string,bool> */
    private $disableEntities = [];

    /**
     * AuditManager constructor.
     *
     * @param EntityFactory<T, IdType> $entityFactory
     * @param ManagerRegistry          $doctrine
     * @param AttributeReader          $reader
     * @param AuditQueue<AuditBuilder> $queue
     */
    public function __construct(
        EntityFactory $entityFactory,
        ManagerRegistry $doctrine,
        AttributeReader $reader,
        AuditQueue $queue
    ) {
        $this->entityFactory   = $entityFactory;
        $this->doctrine        = $doctrine;
        $this->metadataFactory = new AuditMetadataFactory($reader);
        $this->queue           = $queue;
    }

    /**
     * @param EventCriteria $entityCriteria criteria to filter events by
     * @param mixed         $forUser        Filter only events done by specific user
     *
     * @return \Doctrine\ORM\QueryBuilder
     */
    public function findEntityAuditsQb(EventCriteria $entityCriteria, $forUser = null): \Doctrine\ORM\QueryBuilder
    {
        $class = $this->entityFactory->getEventClass();

        /** @var null|EntityManager $em */
        $em = $this->doctrine->getManagerForClass($class);
        if (null === $em) {
            throw $this->noEntityManagerException($class);
        }

        /** @var EventRepository $eventRepo */
        $eventRepo    = new EventRepository($em, $em->getClassMetadata($class));
        $queryBuilder = $eventRepo->match($entityCriteria);

        $orX = $queryBuilder->expr()->orX();

        foreach ($entityCriteria->entities as $index => $entity) {
            if (empty($entity->entityClass)) {
                // must have class to search, can't search by id
                continue;
            }

            $classParam = ':paramClass' . $index;
            $idParam    = ':paramId' . $index;

            $expr = $queryBuilder->expr()->andX('map.entityClass = ' . $classParam);
            $queryBuilder->setParameter($classParam, $entity->entityClass);

            if ($entity->entityId) {
                $expr->add('map.entityId = ' . $idParam);
                $queryBuilder->setParameter($idParam, $entity->entityId);
            }

            $orX->add($expr);
        }

        $queryBuilder->select('evt, change, user')
            ->leftJoin('evt.changes', 'change')
            ->leftJoin('evt.userCreated', 'user');

        if ($orX->count() > 0) {
            $queryBuilder->innerJoin('evt.maps', 'map', Expr\Join::WITH, (string) $orX);
        }

        // TODO review and optimize query if needed
        if (null !== $forUser) {
            $queryBuilder->andWhere('evt.userCreated = :userCreated')->setParameter('userCreated', $forUser);
        }

        return $queryBuilder;
    }

    /**
     * @template C of object
     *
     * @param ClassMetadata<C> $classMetadata
     *
     * @return null|AuditMetadata<C>
     */
    public function getAuditMetadata(ClassMetadata $classMetadata): ?AuditMetadata
    {
        return $this->metadataFactory->getAuditMetadata($classMetadata);
    }

    /**
     * @param object                $entity
     * @param ClassMetadata<object> $metadata
     *
     * @return Closure
     */
    public function getEntityIdClosure($entity, ClassMetadata $metadata): Closure
    {
        $id = $this->getEntityId($entity, $metadata);

        if (null !== $id) {
            // if we have id available right a way return closure which returns that id.
            // during delete operations we have ID now but do not have it later. Updates would work either way
            return function () use ($id) {
                return $id;
            };
        }

        // otherwise return closure which will try to fetch the ID manually.
        // during insert operations we do NOT have the ID now, but we have it later. Updates would work either way
        return function () use ($entity, $metadata) {
            return $this->getEntityId($entity, $metadata);
        };
    }

    /**
     * Creates an empty audit builder.
     *
     * @param null|object $entity only used by internal auditing subscribers; for manual audits this is ignored
     *
     * @return AuditBuilder<T, IdType>
     */
    public function createAuditBuilder($entity = null): AuditBuilder
    {
        return new AuditBuilder($this->entityFactory, $entity);
    }

    /**
     * Creates an audit builder pre-configured for provided entity.
     *
     * Id, alias, event map are audo-populated, you only need to add changes/message.
     *
     * @param int    $type
     * @param object $entity
     *
     * @return AuditBuilder<T, IdType>
     */
    public function createEntityAuditBuilder(int $type, $entity): AuditBuilder
    {
        $builder = $this->createAuditBuilder($entity);
        $class   = \get_class($entity);
        $em      = $this->doctrine->getManagerForClass($class);

        if (null === $em) {
            throw $this->noEntityManagerException($class);
        }

        $classMeta = $em->getClassMetadata($class);
        $auditMeta = $this->getAuditMetadata($classMeta);
        if (null === $auditMeta) {
            throw new InvalidArgumentException(
                \sprintf('Class "%s" is missing "%s" attribute', $class, Apply::class)
            );
        }
        $realClass = $classMeta->getName();

        $builder->eventFor($type, $realClass)
            ->withId($this->getEntityIdClosure($entity, $classMeta))
            ->alias($auditMeta->getAlias())
            ->description($auditMeta->describe($entity));

        return $builder;
    }

    /**
     * Adds the builder to the queue of audits to be persisted.
     *
     * @param AuditBuilder<T, IdType> $builder
     */
    public function enqueueAudit(AuditBuilder $builder): void
    {
        $this->queue->enqueue($builder->validate());
    }

    /**
     * Returns a generator for currently queued audit events.
     *
     * If $class is provided only events for that class are returned.
     *
     * @param null|string $class a class name to filter by
     *
     * @return AuditBuilder<T, IdType>[]|Generator
     */
    public function findQueuedAudits(?string $class = null): Generator
    {
        $iteratorMode = $this->queue->getIteratorMode();

        try {
            $this->queue->setIteratorMode(SplDoublyLinkedList::IT_MODE_KEEP);

            foreach ($this->queue as $item) {
                /** @var AuditBuilder<T, IdType> $item */
                if (null === $class || $class === $item->getEntityClass()) {
                    yield $item;
                }
            }
        } finally {
            $this->queue->setIteratorMode($iteratorMode);
        }
    }

    /**
     * @param AuditBuilder<T, IdType> $builder
     * @param null|object             $entity
     * @param null|ObjectManager      $em
     */
    public function mapEvents(AuditBuilder $builder, $entity, ?ObjectManager $em = null): void
    {
        if (null === $entity) {
            throw new InvalidArgumentException('The "$entity" argument cannot be null.');
        }

        if (null === $em) {
            $em = $this->doctrine->getManagerForClass(\get_class($entity));
        }

        if (null === $em) {
            throw $this->noEntityManagerException(\get_class($entity));
        }

        $mapper = new AuditEventMapper($em, $this);
        $mapper->buildEventMap($builder, $entity);
    }

    /**
     * @param string $entityClass
     *
     * @return bool
     */
    public function isDisableForEntity(string $entityClass): bool
    {
        return isset($this->disableEntities[$entityClass]);
    }

    /**
     * @param string $entityClass
     */
    public function disableForEntity(string $entityClass): void
    {
        if ($this->isDisableForEntity($entityClass)) {
            return;
        }

        $this->disableEntities[$entityClass] = true;
    }

    /**
     * @param string $entityClass
     */
    public function enableForEntity(string $entityClass): void
    {
        if ($this->isDisableForEntity($entityClass)) {
            unset($this->disableEntities[$entityClass]);
        }
    }

    /**
     * @param object                $entity
     * @param ClassMetadata<object> $metadata
     *
     * @return null|int|string
     */
    private function getEntityId($entity, ClassMetadata $metadata)
    {
        $idValues = $metadata->getIdentifierValues($entity);

        if (empty($idValues)) {
            return null;
        }

        $idFields = $metadata->getIdentifierFieldNames();

        if (isset($idFields[1])) { // if more than 1 means composite id
            return \json_encode($idValues) ?: null;
        }

        /** @var scalar|Stringable $id */
        $id = $idValues[$idFields[0]];

        if (\is_int($id)) {
            return $id;
        }

        // if identifier is not composite it means there will be a single id value
        return (string) $id;
    }

    /**
     * @param class-string $class
     *
     * @return InvalidArgumentException
     */
    private function noEntityManagerException(string $class): InvalidArgumentException
    {
        return new InvalidArgumentException(
            \sprintf('Class "%s" is not an entity known to any managers.', $class)
        );
    }
}
