# Tracking Properties in Entities Extending Third-Party Classes

When using the audit bundle, you can control which properties are tracked using the `#[Track]` and `#[Skip]` attributes. However, when your entity extends a class from a third-party library, you can't directly add these attributes to the parent class properties.

The `ParentProps` attribute solves this problem by allowing you to specify which properties from parent classes should be tracked or skipped.

## Usage

Apply the `#[ParentProps]` attribute at the class level to specify which properties from parent classes should be tracked or skipped:

```php
use Cyber\AuditBundle\Attribute as Audit;

#[Audit\Apply]
#[Audit\ParentProps(track: ['name', 'email'], skip: ['password', 'createdAt'])]
class MyEntity extends ThirdPartyEntity
{
    // Your entity properties and methods
}
```

## How It Works

The `ParentProps` attribute accepts two parameters:

- `track`: An array of property names from the parent class that should be tracked. This is useful when the inclusion policy is 'NONE'.
- `skip`: An array of property names from the parent class that should be skipped. This is useful when the inclusion policy is 'ALL'.

### With Inclusion Policy 'ALL'

When the inclusion policy is 'ALL' (default), all properties are tracked by default. In this case:

1. Properties listed in `track` will be tracked (redundant but allowed)
2. Properties listed in `skip` will be skipped
3. All other properties from the parent class will be tracked

```php
#[Audit\Apply] // Default inclusion policy is 'ALL'
#[Audit\ParentProps(skip: ['password', 'createdAt'])]
class MyEntity extends ThirdPartyEntity
{
    // All parent properties except 'password' and 'createdAt' will be tracked
}
```

### With Inclusion Policy 'NONE'

When the inclusion policy is 'NONE', no properties are tracked by default. In this case:

1. Properties listed in `track` will be tracked
2. Properties listed in `skip` will be ignored (redundant but allowed)
3. All other properties from the parent class will not be tracked

```php
#[Audit\Apply(inclusionCriteria: 'NONE')]
#[Audit\ParentProps(track: ['name', 'email'])]
class MyEntity extends ThirdPartyEntity
{
    // Only 'name' and 'email' from the parent class will be tracked
}
```

## Example

```php
use Cyber\AuditBundle\Attribute as Audit;
use ThirdParty\UserEntity;

#[Audit\Apply]
#[Audit\ParentProps(
    track: ['firstName', 'lastName', 'email'],
    skip: ['password', 'securityToken', 'lastLoginAt']
)]
class User extends UserEntity
{
    #[Audit\Track]
    private string $role;

    // The parent properties 'firstName', 'lastName', and 'email' will be tracked
    // The parent properties 'password', 'securityToken', and 'lastLoginAt' will be skipped
    // All other parent properties will be tracked (because inclusion policy is 'ALL')
    // The 'role' property in this class will be tracked
}
```
