<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Cyber\AuditBundle\Attribute\Apply;
use Cyber\AuditBundle\Attribute\Cascade;
use Cyber\AuditBundle\Attribute\Describe;
use Cyber\AuditBundle\Attribute\ParentProps;
use Cyber\AuditBundle\Attribute\Skip;
use Cyber\AuditBundle\Attribute\Track;
use Cyber\AuditBundle\Describer\MethodDescriber;
use Cyber\AuditBundle\Describer\PropertyDescriber;
use Cyber\AuditBundle\Service\AttributeReader;
use Cyber\OrmExtras\Utility\SoftDeletable;
use Doctrine\Persistence\Mapping\ClassMetadata;
use InvalidArgumentException;
use Psr\Cache\CacheItemPoolInterface;
use ReflectionClass;
use ReflectionProperty;

/**
 * @internal
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class AuditMetadataFactory
{
    /**
     * @var array<string, AuditMetadata<object>|false>
     */
    private $auditMetadata = [];

    /** @var AttributeReader */
    private $reader;

    /** @var null|CacheItemPoolInterface */
    private $cache;

    public function __construct(AttributeReader $reader, ?CacheItemPoolInterface $cache = null)
    {
        $this->reader = $reader;
        $this->cache  = $cache;
    }

    /**
     * @template T of object
     *
     * @param ClassMetadata<T> $classMetadata
     *
     * @return null|AuditMetadata<T>
     */
    public function getAuditMetadata(ClassMetadata $classMetadata): ?AuditMetadata
    {
        $reflection = $classMetadata->getReflectionClass();
        $className  = $reflection->getName();
        $cacheKey   = $this->sanitizeCacheKey($className);

        // Try to get from cache first
        $cachedMetadata = $this->getFromCache($className, $cacheKey);
        if (null !== $cachedMetadata) {
            return $cachedMetadata;
        }

        /** @var null|Apply $audit */
        $audit = $this->reader->getClassAttribute($reflection, Apply::class);
        if (null === $audit) {
            $this->storeInCache($className, $cacheKey, false);

            return null;
        }

        $metadata = $this->createMetadata($classMetadata, $reflection, $audit);
        $this->storeInCache($className, $cacheKey, $metadata);

        return $metadata;
    }

    /**
     * @template T of object
     *
     * @param class-string<T> $className
     * @param string          $cacheKey
     *
     * @return null|AuditMetadata<T>
     */
    private function getFromCache(string $className, string $cacheKey): ?AuditMetadata
    {
        // Check in-memory cache first
        if (isset($this->auditMetadata[$className])) {
            /** @var AuditMetadata<T>|false $meta */
            $meta = $this->auditMetadata[$className];

            // convert false to null, false means item was checked and did not have metadata
            return $meta ?: null;
        }

        // Try to get from Symfony cache if available
        if (null !== $this->cache) {
            $cacheItem = $this->cache->getItem($cacheKey);
            if ($cacheItem->isHit()) {
                $meta = $cacheItem->get();
                if (!$meta instanceof AuditMetadata) {
                    // outdated or corrupted cache, return null to regenerate
                    return null;
                }
                // Store in memory cache for faster subsequent access
                $this->auditMetadata[$className] = $meta;

                return $meta;
            }
        }

        return null;
    }

    /**
     * @template T of object
     *
     * @param string                 $className
     * @param string                 $cacheKey
     * @param AuditMetadata<T>|false $metadata
     */
    private function storeInCache(string $className, string $cacheKey, $metadata): void
    {
        // Store in memory cache
        $this->auditMetadata[$className] = $metadata;

        // Store in Symfony cache if available
        if (null !== $this->cache) {
            $cacheItem = $this->cache->getItem($cacheKey);
            $cacheItem->set($metadata);
            $this->cache->save($cacheItem);
        }
    }

    /**
     * @template T of object
     *
     * @param ClassMetadata<T>   $classMetadata
     * @param ReflectionClass<T> $reflection
     * @param Apply              $audit
     *
     * @return AuditMetadata<T>
     */
    private function createMetadata(
        ClassMetadata $classMetadata,
        ReflectionClass $reflection,
        Apply $audit
    ): AuditMetadata {
        $metadata = new AuditMetadata($classMetadata);
        $metadata->setAlias($audit->displayName ?? $reflection->getShortName());

        $this->processProperties($metadata, $reflection, Apply::INCLUDE_ALL === $audit->inclusionCriteria);

        // Process parent properties if the ParentProps attribute is present
        $parentProps = $this->reader->getClassAttribute($reflection, ParentProps::class);
        if (null !== $parentProps) {
            $this->processParentProperties(
                $metadata,
                $reflection,
                $parentProps,
                Apply::INCLUDE_ALL === $audit->inclusionCriteria
            );
        }

        foreach ($reflection->getMethods() as $method) {
            $annot = $this->reader->getMethodAttribute($method, Describe::class);
            if (null !== $annot) {
                $metadata->setDescriber(new MethodDescriber($method->getName()));
            }
        }

        // check if entity is soft deleteable and add tracking flag to metadata
        if ($reflection->implementsInterface(SoftDeletable::class)) {
            $metadata->trackSoftDelete();
        }

        $metadata->compile();

        return $metadata;
    }

    /**
     * @param AuditMetadata<object>   $metadata
     * @param ReflectionClass<object> $reflection
     * @param bool                    $all
     */
    private function processProperties(AuditMetadata $metadata, ReflectionClass $reflection, bool $all): void
    {
        $classMeta = $metadata->getClassMetadata();

        foreach ($reflection->getProperties() as $property) {
            [$track, $skip] = $this->processProperty($metadata, $property);

            if ($track && $skip) {
                throw new InvalidArgumentException(\sprintf(
                    'You have set both Track and Skip attribute on %s::$%s, please pick only one.',
                    $reflection->getName(),
                    $property->getName()
                ));
            }

            if ($track || ($all && !$skip)) {
                $name = $property->getName();
                $this->addPropertyToMetadata($metadata, $classMeta, $name);
            }
        }
    }

    /**
     * @param AuditMetadata<object> $metadata
     * @param ClassMetadata         $classMeta
     * @param string                $name
     */
    private function addPropertyToMetadata(AuditMetadata $metadata, ClassMetadata $classMeta, string $name): void
    {
        $metadata->addProperty($name);

        $embedded = $classMeta->embeddedClasses[$name] ?? null;
        if ($embedded && $embedded['class']) {
            $metadata->setEmbeddedClass($name, $embedded['class']);
        }
    }

    /**
     * Sanitizes a cache key to be compatible with PSR-6 cache and adds the required suffix.
     *
     * @param string $key The original cache key
     *
     * @return string The sanitized cache key with suffix
     */
    private function sanitizeCacheKey(string $key): string
    {
        // Replace characters that might be problematic in cache keys
        $sanitized = \preg_replace('/[^A-Za-z0-9_.]/', '_', $key);

        // Add the required suffix
        return $sanitized . '__CYBER_AUDIT_META__';
    }

    /**
     * @param AuditMetadata<object> $metadata
     * @param ReflectionProperty    $property
     *
     * @return bool[]
     */
    private function processProperty(AuditMetadata $metadata, ReflectionProperty $property): array
    {
        $track = false;
        $skip  = false;

        foreach ($this->reader->getPropertyAttributes($property) as $annot) {
            switch (true) {
                case $annot instanceof Cascade:
                    $metadata->addOwner($property);
                    break;
                case $annot instanceof Track:
                    $track = true;
                    if (null !== $annot->userValueList) {
                        $valueOptions = \is_array($annot->userValueList) ? $annot->userValueList : \constant($annot->userValueList);
                        \assert(
                            \is_array($valueOptions),
                            \sprintf(
                                'User value list for property "%s" must be an array or constant that resolves to an array.',
                                $property->getName()
                            )
                        );
                        $metadata->setUserValues($property->getName(), $valueOptions);
                    }
                    break;
                case $annot instanceof Skip:
                    $skip = true;
                    break;
                case $annot instanceof Describe:
                    $metadata->setDescriber(new PropertyDescriber($property));
                    break;
            }
        }

        return [$track, $skip];
    }

    /**
     * Process properties from parent classes based on the ParentProps attribute.
     *
     * @param AuditMetadata<object>   $metadata
     * @param ReflectionClass<object> $reflection
     * @param ParentProps             $parentProps
     * @param bool                    $all         Whether the inclusion policy is ALL
     */
    private function processParentProperties(
        AuditMetadata $metadata,
        ReflectionClass $reflection,
        ParentProps $parentProps,
        bool $all
    ): void {
        $parent = $reflection->getParentClass();
        if (!$parent) {
            return;
        }

        $classMeta = $metadata->getClassMetadata();

        // Process properties to track
        foreach ($parentProps->track as $propName) {
            if (!$parent->hasProperty($propName)) {
                continue;
            }
            $this->addPropertyToMetadata($metadata, $classMeta, $propName);
        }

        if ($all) {

            // If inclusion policy is ALL, process properties to skip
            foreach ($parent->getProperties() as $property) {
                $name = $property->getName();
                if (\in_array($name, $parentProps->skip, true) || \in_array($name, $parentProps->track, true)) {
                    // if skipped or already tracked from first loop, continue
                    continue;
                }

                $this->addPropertyToMetadata($metadata, $classMeta, $name);
            }
        }
    }
}
