<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle;

use Cyber\AuditBundle\AuditMetadata;
use Cyber\AuditBundle\Describer\Describer;
use Cyber\AuditBundle\Describer\ToStringDescriber;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping\ClassMetadata;
use InvalidArgumentException;
use LogicException;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use ReflectionProperty;
use Tests\Cyber\AuditBundle\Mock\MockComment;
use Tests\Cyber\AuditBundle\Mock\MockUser;

/**
 * @internal
 *
 * @covers \Cyber\AuditBundle\AuditMetadata
 */
class AuditMetadataTest extends TestCase
{
    private ClassMetadata $classMetadata;

    /** @var AuditMetadata<object> */
    private AuditMetadata $auditMetadata;

    private ReflectionProperty $reflectionProperty;

    protected function setUp(): void
    {
        // mock actual class as we need __sleep to test serialization
        $this->classMetadata = $this->createMock(ClassMetadata::class);
        $this->classMetadata->method('getReflectionClass')
            ->willReturn(new ReflectionClass(MockUser::class));
        $this->classMetadata->method('getName')
            ->willReturn(MockUser::class);
        $this->classMetadata->method('__sleep')->willReturn([]);

        // Create the AuditMetadata instance
        $this->auditMetadata = new AuditMetadata($this->classMetadata);

        // Create a ReflectionProperty for testing
        $this->reflectionProperty = new ReflectionProperty(MockUser::class, 'name');
        $this->reflectionProperty->setAccessible(true);
    }

    public function testAddAndGetOwners(): void
    {
        // Add an owner
        $result = $this->auditMetadata->addOwner($this->reflectionProperty);

        // Assert fluent interface returns $this
        static::assertSame($this->auditMetadata, $result);

        // Get owners and verify
        $owners = $this->auditMetadata->getOwners();
        static::assertInstanceOf(ArrayCollection::class, $owners);
        static::assertCount(1, $owners);
        static::assertSame($this->reflectionProperty, $owners['name']);
    }

    public function testRemoveOwner(): void
    {
        // Add and then remove an owner
        $this->auditMetadata->addOwner($this->reflectionProperty);
        $this->auditMetadata->removeOwner($this->reflectionProperty);

        // Verify owner was removed
        $owners = $this->auditMetadata->getOwners();
        static::assertCount(0, $owners);
    }

    public function testAddAndGetProperties(): void
    {
        // Add a property
        $result = $this->auditMetadata->addProperty('name');

        // Assert fluent interface returns $this
        static::assertSame($this->auditMetadata, $result);

        // Get properties and verify
        $properties = $this->auditMetadata->getProperties();
        static::assertInstanceOf(ArrayCollection::class, $properties);
        static::assertCount(1, $properties);
        static::assertTrue($properties->contains('name'));
    }

    public function testRemoveProperty(): void
    {
        // Add and then remove a property
        $this->auditMetadata->addProperty('name');
        $this->auditMetadata->removeProperty('name');

        // Verify property was removed
        $properties = $this->auditMetadata->getProperties();
        static::assertCount(0, $properties);
    }

    public function testSetAndGetUserValues(): void
    {
        $userValues = ['admin', 'user'];

        // Set user values
        $result = $this->auditMetadata->setUserValues('role', $userValues);

        // Assert fluent interface returns $this
        static::assertSame($this->auditMetadata, $result);

        // Get user values and verify
        $retrievedValues = $this->auditMetadata->getUserValues('role');
        static::assertSame($userValues, $retrievedValues);

        // Test getting non-existent user values
        static::assertNull($this->auditMetadata->getUserValues('nonexistent'));
    }

    public function testSetAndGetEmbeddedClass(): void
    {
        // Set embedded class
        $result = $this->auditMetadata->setEmbeddedClass('address', MockComment::class);

        // Assert fluent interface returns $this
        static::assertSame($this->auditMetadata, $result);

        // Check if field is embedded
        static::assertTrue($this->auditMetadata->isEmbedded('address'));
        static::assertFalse($this->auditMetadata->isEmbedded('nonexistent'));

        // Get embedded class and verify
        $embeddedClass = $this->auditMetadata->getEmbeddedClass('address');
        static::assertSame(MockComment::class, $embeddedClass);
    }

    public function testGetEmbeddedClassThrowsExceptionForNonEmbeddedField(): void
    {
        $this->expectException(InvalidArgumentException::class);
        $this->auditMetadata->getEmbeddedClass('nonexistent');
    }

    public function testGetClassMetadata(): void
    {
        $metadata = $this->auditMetadata->getClassMetadata();
        static::assertSame($this->classMetadata, $metadata);
    }

    public function testTrackSoftDelete(): void
    {
        // Initially soft delete tracking should be false
        static::assertFalse($this->auditMetadata->isTrackSoftDelete());

        // Enable soft delete tracking
        $this->auditMetadata->trackSoftDelete();

        // Verify soft delete tracking is enabled
        static::assertTrue($this->auditMetadata->isTrackSoftDelete());
    }

    public function testSetAndGetAlias(): void
    {
        $alias = 'user_entity';

        // Set alias
        $result = $this->auditMetadata->setAlias($alias);

        // Assert fluent interface returns $this
        static::assertSame($this->auditMetadata, $result);

        // Get alias and verify
        $retrievedAlias = $this->auditMetadata->getAlias();
        static::assertSame($alias, $retrievedAlias);
    }

    public function testSetAndGetDescriber(): void
    {
        $describer = $this->createMock(Describer::class);

        // Set describer
        $this->auditMetadata->setDescriber($describer);

        // Get describer and verify
        $retrievedDescriber = $this->auditMetadata->getDescriber();
        static::assertSame($describer, $retrievedDescriber);
    }

    public function testSetDescriberThrowsExceptionWhenAlreadySet(): void
    {
        $describer1 = $this->createMock(Describer::class);
        $describer2 = $this->createMock(Describer::class);

        // Set first describer
        $this->auditMetadata->setDescriber($describer1);

        // Attempt to set second describer should throw exception
        $this->expectException(LogicException::class);
        $this->auditMetadata->setDescriber($describer2);
    }

    public function testDescribe(): void
    {
        $mockUser = new MockUser();
        $mockUser->setName('John Doe');

        // Initially there's no describer
        static::assertFalse($this->auditMetadata->canDescribe());
        static::assertNull($this->auditMetadata->describe($mockUser));

        // Set a describer
        $describer = $this->createMock(Describer::class);
        $describer->method('describe')->willReturn('John Doe');
        $this->auditMetadata->setDescriber($describer);

        // Now we can describe
        static::assertTrue($this->auditMetadata->canDescribe());
        static::assertSame('John Doe', $this->auditMetadata->describe($mockUser));
    }

    public function testCompileWithToString(): void
    {
        // Compile should set ToStringDescriber if __toString exists and no describer is set
        $this->auditMetadata->compile();

        // Verify a ToStringDescriber was set
        static::assertInstanceOf(ToStringDescriber::class, $this->auditMetadata->getDescriber());
    }

    public function testSerializationAndDeserialization(): void
    {
        // Set up the AuditMetadata with various properties
        $this->auditMetadata->addOwner($this->reflectionProperty);
        $this->auditMetadata->addProperty('name');
        $this->auditMetadata->setUserValues('role', ['admin', 'user']);
        $this->auditMetadata->setEmbeddedClass('address', MockComment::class);
        $this->auditMetadata->trackSoftDelete();
        $this->auditMetadata->setAlias('user_entity');

        // Serialize and deserialize
        $serialized   = \serialize($this->auditMetadata);
        $deserialized = \unserialize($serialized);

        // Verify deserialized object has the same properties
        static::assertInstanceOf(AuditMetadata::class, $deserialized);
        static::assertTrue($deserialized->getProperties()->contains('name'));
        static::assertSame(['admin', 'user'], $deserialized->getUserValues('role'));
        static::assertTrue($deserialized->isEmbedded('address'));
        static::assertSame(MockComment::class, $deserialized->getEmbeddedClass('address'));
        static::assertTrue($deserialized->isTrackSoftDelete());
        static::assertSame('user_entity', $deserialized->getAlias());

        // Verify owners were reconstructed
        $owners = $deserialized->getOwners();
        static::assertCount(1, $owners);
        static::assertInstanceOf(ReflectionProperty::class, $owners['name']);
        static::assertSame('name', $owners['name']->getName());
    }
}
