<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\AuditBundle\Service;

use ArrayObject;
use Attribute;
use Cyber\AuditBundle\Attribute\AuditAttribute;
use LogicException;
use ReflectionAttribute;
use ReflectionClass;
use ReflectionMethod;
use ReflectionProperty;

/**
 * @internal
 */
final class AttributeReader
{
    /** @var array<class-string<AuditAttribute>,bool> */
    private array $isRepeatable = [];

    /**
     * @template T of AuditAttribute
     *
     * @return T[]
     */
    public function getClassAttributes(ReflectionClass $class): array
    {
        return $this->convertToAttributeInstances($class->getAttributes());
    }

    /**
     * @template T of AuditAttribute
     */
    public function getMethodAttributes(ReflectionMethod $method): array
    {
        return $this->convertToAttributeInstances($method->getAttributes());
    }

    /**
     * @template T of AuditAttribute
     */
    public function getPropertyAttributes(ReflectionProperty $property): array
    {
        return $this->convertToAttributeInstances($property->getAttributes());
    }

    /**
     * @param class-string<T> $attributeClass the name of the attribute
     *
     * @return null|T
     *
     * @template T of AuditAttribute
     */
    public function getPropertyAttribute(ReflectionProperty $property, string $attributeClass)
    {
        if ($this->isRepeatable($attributeClass)) {
            throw new LogicException(\sprintf(
                'The attribute "%s" is repeatable. Call getPropertyAttributeCollection() instead.',
                $attributeClass
            ));
        }

        return $this->getPropertyAttributes($property)[$attributeClass] ?? null;
    }

    /**
     * @template T of object
     *
     * @param ReflectionClass<object> $class
     * @param class-string<T>         $attributeClass
     *
     * @return null|T
     */
    public function getClassAttribute(ReflectionClass $class, string $attributeClass)
    {
        $attributes = $this->getClassAttributes($class);

        foreach ($attributes as $attribute) {
            if ($attribute instanceof $attributeClass) {
                return $attribute;
            }
        }

        return null;
    }

    public function getMethodAttribute(ReflectionMethod $method, $attributeClass)
    {
        $attributes = $this->getMethodAttributes($method);

        foreach ($attributes as $attribute) {
            if ($attribute instanceof $attributeClass) {
                return $attribute;
            }
        }

        return null;
    }

    /**
     * @param class-string<T> $attributeClass the name of the attribute
     *
     * @return ArrayObject<T>
     *
     * @template T of AuditAttribute
     */
    public function getPropertyAttributeCollection(
        ReflectionProperty $property,
        string $attributeClass
    ): ArrayObject {
        if (!$this->isRepeatable($attributeClass)) {
            throw new LogicException(\sprintf(
                'The attribute "%s" is not repeatable. Call getPropertyAttribute() instead.',
                $attributeClass
            ));
        }

        return $this->getPropertyAttributes($property)[$attributeClass] ?? new ArrayObject();
    }

    /**
     * @param array<ReflectionAttribute> $attributes
     *
     * @template T of AuditAttribute
     */
    private function convertToAttributeInstances(array $attributes): array
    {
        $instances = [];

        foreach ($attributes as $attribute) {
            $attributeName = $attribute->getName();
            \assert(\is_string($attributeName));
            // Make sure we only get our Annotations
            if (!\is_subclass_of($attributeName, AuditAttribute::class)) {
                continue;
            }

            $instance = $attribute->newInstance();
            \assert($instance instanceof AuditAttribute);

            if ($this->isRepeatable($attributeName)) {
                if (!isset($instances[$attributeName])) {
                    $instances[$attributeName] = new ArrayObject();
                }

                $collection = $instances[$attributeName];
                \assert($collection instanceof ArrayObject);
                $collection[] = $instance;
                continue;
            }

            $instances[$attributeName] = $instance;
        }

        return $instances;
    }

    /**
     * @param class-string<AuditAttribute> $attributeClassName
     */
    private function isRepeatable(string $attributeClassName): bool
    {
        if (isset($this->isRepeatable[$attributeClassName])) {
            return $this->isRepeatable[$attributeClassName];
        }

        $reflectionClass = new ReflectionClass($attributeClassName);
        /** @var Attribute $attribute */
        $attribute = $reflectionClass->getAttributes()[0]->newInstance();

        return $this->isRepeatable[$attributeClassName] = ($attribute->flags & Attribute::IS_REPEATABLE) > 0;
    }
}
