<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\AuditBundle;

use Symfony\Component\Config\Definition\Configurator\DefinitionConfigurator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Loader\Configurator\ContainerConfigurator;
use Symfony\Component\HttpKernel\Bundle\AbstractBundle;

class CyberAuditBundle extends AbstractBundle
{
    public function configure(DefinitionConfigurator $definition): void
    {
        // @formatter:off
        $definition->rootNode()
            ->children()
                ->enumNode('metadata_driver')
                    ->values(['attribute'])
                    ->defaultValue('attribute')
                    ->info('What will you use to provide Audit configs to entities')
                ->end()
                ->arrayNode('entity')
                    ->children()
                        ->scalarNode('change')
                            ->isRequired()
                            ->cannotBeEmpty()
                            ->info('Class name of the Change entity')
                        ->end()
                        ->scalarNode('event')
                            ->isRequired()
                            ->cannotBeEmpty()
                            ->info('Class name of the Event entity')
                        ->end()
                        ->scalarNode('map')
                            ->isRequired()
                            ->cannotBeEmpty()
                            ->info('Class name of the EventMap entity')
                        ->end()
                        ->scalarNode('factory')
                            ->defaultNull()
                            ->info('Custom Entity Factory class or @service')
                        ->end()
                    ->end()
                ->end()
            ->end();
        // @formatter:on
    }

    /**
     * @param array{metadata_driver?: string, entity?: array<string, string>} $config
     */
    public function loadExtension(array $config, ContainerConfigurator $container, ContainerBuilder $builder): void
    {
        if (!isset($config['entity'])) {
            // if we don't have entity config, do not load anything.
            // user probably added the bundle but was not ready to configure it yet.
            return;
        }

        $container->import('../config/services.php');

        // Load debug services only when kernel.debug is true
        if ($builder->getParameter('kernel.debug')) {
            $container->import('../config/services_dev.php');
        }

        foreach ($config['entity'] as $key => $value) {
            $container->parameters()->set('cyber_audit.entity.' . $key, $value);
        }

        $factory = $config['entity']['factory'] ?? null;
        if ($factory) {
            if (0 === \mb_strpos($factory, '@')) {
                $container->services()->alias('cyber_audit.service.entity_factory', \mb_substr($factory, 1));

                return;
            }
            $builder->getDefinition('cyber_audit.service.entity_factory')
                ->setClass($factory);
        }
    }
}
