<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\AuditBundle;

use Cyber\AuditBundle\CyberAuditBundle;
use Cyber\AuditBundle\Service\EntityFactory;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBag;

/**
 * @internal
 *
 * @covers \Cyber\AuditBundle\CyberAuditBundle
 */
class CyberAuditTest extends TestCase
{
    public function testEmptyLoad(): void
    {
        $bundle    = new CyberAuditBundle();
        $extension = $bundle->getContainerExtension();
        $container = new ContainerBuilder(new ParameterBag([
            'kernel.environment' => 'test',
            'kernel.build_dir'   => __DIR__,
        ]));

        static::assertNotNull($extension, 'Extension should not be null');
        $extension->load([[]], $container);

        static::assertFalse($container->hasDefinition('cyber_audit.event_listener.kernel'));
    }

    public function testMinimalConfigs(): void
    {
        $bundle    = new CyberAuditBundle();
        $extension = $bundle->getContainerExtension();
        $container = new ContainerBuilder(new ParameterBag([
            'kernel.environment' => 'test',
            'kernel.build_dir'   => __DIR__,
            'kernel.debug'       => true,
        ]));

        $config = [
            'entity' => [
                'change' => 'C',
                'event'  => 'E',
                'map'    => 'M',
            ],
        ];

        static::assertNotNull($extension, 'Extension should not be null');
        $extension->load([$config], $container);

        $factory = $container->get('cyber_audit.service.entity_factory');
        static::assertInstanceOf(EntityFactory::class, $factory);
    }

    public function testCustomFactory(): void
    {
        $bundle    = new CyberAuditBundle();
        $extension = $bundle->getContainerExtension();
        $container = new ContainerBuilder(new ParameterBag([
            'kernel.environment' => 'test',
            'kernel.build_dir'   => __DIR__,
            'kernel.debug'       => true,
        ]));

        $config = [
            'entity' => [
                'change'  => 'C',
                'event'   => 'E',
                'map'     => 'M',
                'factory' => DecoratedEntityFactory::class,
            ],
        ];

        static::assertNotNull($extension, 'Extension should not be null');
        $extension->load([$config], $container);

        $factory = $container->get('cyber_audit.service.entity_factory');
        static::assertInstanceOf(DecoratedEntityFactory::class, $factory);
        static::assertEquals(['C', 'E', 'M'], $factory->getParams());

        $bundle    = new CyberAuditBundle();
        $extension = $bundle->getContainerExtension();
        $container = new ContainerBuilder(new ParameterBag([
            'kernel.environment' => 'test',
            'kernel.build_dir'   => __DIR__,
            'kernel.debug'       => true,
        ]));
        $container->register(DecoratedEntityFactory::class, DecoratedEntityFactory::class)
            ->setArguments(['X', 'Y', 'Z']);

        $config = [
            'entity' => [
                'change'  => 'C',
                'event'   => 'E',
                'map'     => 'M',
                'factory' => '@' . DecoratedEntityFactory::class,
            ],
        ];

        static::assertNotNull($extension, 'Extension should not be null');
        $extension->load([$config], $container);

        $factory = $container->get('cyber_audit.service.entity_factory');
        static::assertInstanceOf(DecoratedEntityFactory::class, $factory);
        static::assertEquals(['X', 'Y', 'Z'], $factory->getParams());
    }
}
