<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\AuditBundle\Service;

use Cyber\AuditBundle\AuditBuilder;
use Cyber\AuditBundle\Entity\Change;
use Cyber\AuditBundle\Entity\Event;
use Cyber\AuditBundle\Entity\EventMap;
use Cyber\AuditBundle\Service\EntityFactory;
use Generator;
use LogicException;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;

/**
 * @internal
 *
 * @covers \Cyber\AuditBundle\AuditBuilder
 */
class AuditBuilderTest extends TestCase
{
    /** @var AuditBuilder<mixed, mixed> */
    private $builder;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        /** @var EntityFactory<mixed,mixed>|MockObject $factory */
        $factory = $this->getMockBuilder(EntityFactory::class)->disableOriginalConstructor()->getMock();

        $factory->expects(static::any())
            ->method('createEvent')
            ->willReturnCallback(function () {
                return new class () extends Event {
                };
            });

        $factory->expects(static::any())
            ->method('createChange')
            ->willReturnCallback(function () {
                return new class () extends Change {
                };
            });

        $factory->expects(static::any())
            ->method('createMap')
            ->willReturnCallback(function ($event) {
                return new class ($event) extends EventMap {
                };
            });

        $this->builder = new AuditBuilder($factory);
    }

    /**
     * @param string $call
     * @param int    $type
     *
     * @dataProvider eventTypeProvider
     */
    public function testBuildCreateEvent($call, $type): void
    {
        /** @var AuditBuilder<mixed, mixed> $builder */
        $builder = $this->builder->{$call}('Some\FQCN');

        /** @var Event<mixed, mixed> $event */
        $event = $builder
            ->withId('7')
            ->alias('Patient')
            ->description('John Smith')
            ->changed('user', '77')
            ->mappedTo(self::class, '44')
            ->message('Good day')
            ->getEvent();

        static::assertInstanceOf(Event::class, $event);

        static::assertEquals($type, $event->getType());
        static::assertEquals('Some\FQCN', $event->getEntityClass());
        static::assertEquals('Patient', $event->getEntityAlias());
        static::assertEquals(7, $event->getEntityId());
        static::assertEquals('John Smith', $event->getDescription());
        static::assertEquals('Good day', $event->getMessage());

        $changes = $event->getChanges();
        static::assertCount(1, $changes);
        /** @var Change $change */
        $change = $changes[0];
        static::assertEquals('user', $change->getField());
        static::assertEquals('77', $change->getNewValue()->getRawValue());

        $maps = $event->getMaps();
        static::assertCount(1, $maps);
        /** @var EventMap<mixed, mixed> $map */
        $map = $maps[0];
        static::assertEquals(self::class, $map->getEntityClass());
        static::assertEquals(44, $map->getEntityId());
    }

    /**
     * @return Generator<array<mixed>>
     */
    public function eventTypeProvider(): Generator
    {
        yield ['created', Event::TYPE_INSERT];
        yield ['updated', Event::TYPE_UPDATE];
        yield ['deleted', Event::TYPE_DELETE];
        yield ['softDeleted', Event::TYPE_SOFT_DELETE];
        yield ['messageFor', Event::TYPE_MESSAGE];
    }

    /**
     * @group        debug
     */
    public function testValidateValidBuilder(): void
    {
        $result = $this->builder->created('someclass')
            ->alias('somealias')
            ->withId('someid')
            ->mappedTo(self::class, 'someid')
            ->validate();

        static::assertSame($this->builder, $result);
    }

    /**
     * @dataProvider invalidBuilderData
     *
     * @param array<string, mixed[]> $data
     */
    public function testValidateInvalidBuilder($data): void
    {
        $this->expectException(LogicException::class);

        foreach ($data as $method => $arguments) {
            $callable = [$this->builder, $method];
            static::assertIsCallable($callable);
            \call_user_func_array($callable, $arguments);
        }
        $this->builder->validate();
        $this->builder->validateMapping();
    }

    /**
     * @return Generator<array<mixed>>
     */
    public function invalidBuilderData(): Generator
    {
        yield [[]];
        yield [['ofType' => [0]]];
        yield [['created' => ['classname']]];
        yield [
            [
                'created' => ['classname'],
                'alias'   => ['somealias'],
            ],
        ];
        yield [
            [
                'created' => ['classname'],
                'alias'   => ['somealias'],
                'withId'  => ['someid'],
            ],
        ];
    }
}
