<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CacheBundle\Engine\Memcache;

/**
 * Class MemcacheList.
 *
 * @SuppressWarnings(PHPMD.CamelCaseParameterName)
 * @SuppressWarnings(PHPMD.ExcessiveClassLength)
 * @SuppressWarnings(PHPMD.ElseExpression)
 */
class MemcacheList
{
    /** @var PrefixedMemcache */
    private $memcache;

    private $id;

    private $headKey;

    private $current;

    private $tailKey;

    private $meta;

    public function __construct(PrefixedMemcache $memcache, string $key)
    {
        $this->memcache = $memcache;

        $this->id      = 'cyber_list[' . $key . ']';
        $this->headKey = $this->id . '[head]';
        $this->current = $this->id . '[current]';
        $this->tailKey = $this->id . '[tail]';
        $this->meta    = $this->id . '[meta]';
    }

    /**
     * Adds new data to the queue.
     *
     * @param mixed $data the data to be added to queue
     *
     * @return bool|int new size of the list; FALSE on failure
     */
    public function push($data)
    {
        $nextId = $this->getTail(true);
        if (false === $nextId) {
            //key does not exist lets try to add it
            if (false === $this->memcache->add($this->tailKey, 1)) {
                //failed to add maybe another script simultaneously create it try increment again
                $nextId = $this->getTail(true);
                //now definately failure
                if (false === $nextId) {
                    return false;
                }
            } else {
                $nextId = 1;
                $this->memcache->add($this->headKey, 0);
                $this->resetCurrent(0);
            }
        }
        $id = $this->itemKey($nextId - 1);

        if (false === $this->memcache->add($id, $data)) {
            return false;
        }
        $head = $this->getHead();

        return $nextId - $head;
    }

    /**
     * @return array<mixed>|false|string data of the next element in the queue; FALSE if queue is empty or an error occurs
     */
    public function pop()
    {
        $tail = $this->getTail();
        $id   = $this->getHead(true);
        if (false === $id) {
            return false;
        }
        if ($id > $tail) {
            $this->memcache->decrement($this->headKey);

            return false;
        }

        $key  = $this->itemKey($id - 1);
        $data = $this->memcache->get($key);
        $this->memcache->delete($key);

        return $data;
    }

    /**
     * @param int $index
     *
     * @return mixed
     */
    public function get(int $index)
    {
        $ref = $index < 0 ? $this->getTail() : $this->getHead();
        $key = $this->itemKey($ref + $index);

        return $this->memcache->get($key);
    }

    /**
     * @param int   $index
     * @param mixed $value
     *
     * @return bool
     */
    public function set(int $index, $value): bool
    {
        $tail = $this->getTail();
        $head = $this->getHead();

        $ref = $index < 0 ? $tail : $head;
        $id  = $ref + $index;

        if ($id < $head || $id >= $tail) {
            // out of bounds.
            return false;
        }

        $key = $this->itemKey($id);

        return $this->memcache->set($key, $value);
    }

    /**
     * @return mixed
     */
    public function current()
    {
        $id = $this->getCurrent();
        if (false === $id) {
            return false;
        }

        $tail = $this->getTail();
        $head = $this->getHead();

        if ($id < $head || $id >= $tail) {
            $this->resetCurrent($head);
            $id = $this->getCurrent();
            if (false === $id) {
                return false;
            }
        }

        $key = $this->itemKey($id);

        return $this->memcache->get($key);
    }

    /**
     * Returns the current item and moves the pointer to the next item.
     *
     * @return array<mixed>|false|string
     */
    public function next()
    {
        $tail = $this->getTail();
        $head = $this->getHead();
        if (false === $tail || false === $head || 0 === ($tail - $head)) {
            return false;
        }

        $id = $this->getCurrent(true);
        if (false === $id) {
            return false;
        }

        if ($id > $tail) { // we reached the end
            $this->resetCurrent($head);

            return $this->next();
        }

        if ($id === $tail) {
            // we know we are in the end so reset it here as well.
            $this->resetCurrent($head);
        }

        $key = $this->itemKey($id - 1);

        return $this->memcache->get($key);
    }

    /**
     * @return int size of the queue
     */
    public function getSize(): int
    {
        $head = $this->getHead();
        $tail = $this->getTail();
        if (false === $head || false === $tail) {
            return 0;
        }

        return $tail - $head;
    }

    /**
     * Removes all data from the queue;.
     */
    public function clear(): bool
    {
        $head = $this->getHead();
        $tail = $this->getTail();
        if (false === $head || false === $tail) {
            return false;
        }
        $this->memcache->delete($this->tailKey);
        $this->memcache->delete($this->headKey);
        $this->memcache->delete($this->current);
        $this->memcache->delete($this->meta);
        for ($i = $head; $i < $tail; ++$i) {
            $key = $this->itemKey($i);
            $this->memcache->delete($key);
        }

        return true;
    }

    private function itemKey(int $index): string
    {
        return $this->id . '[' . ($index) . ']';
    }

    /**
     * @param bool $increment
     *
     * @return false|int
     * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
     */
    private function getHead($increment = false)
    {
        return $this->getIndex($this->headKey, $increment);
    }

    /**
     * @param bool $increment
     *
     * @return false|int
     * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
     */
    private function getCurrent($increment = false)
    {
        return $this->getIndex($this->current, $increment);
    }

    /**
     * @param mixed $value
     */
    private function resetCurrent($value): void
    {
        $this->memcache->set($this->current, $value);
    }

    /**
     * @param bool $increment
     *
     * @return bool|int
     * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
     */
    private function getTail($increment = false)
    {
        return $this->getIndex($this->tailKey, $increment);
    }

    /**
     * @param string $key
     * @param bool   $increment
     *
     * @return false|int
     */
    private function getIndex(string $key, bool $increment)
    {
        if ($increment) {
            return $this->memcache->increment($key);
        }
        $value = $this->memcache->get($key);
        if (false === $value || \is_array($value)) {
            return false;
        }

        return (int) $value;
    }
}
