# Purpose
Provides a wrapper around various cache engines (such as `Memcache`, `Redis`) with a prefix functionality. Basically 
lets you isolate different parts of your application under different prefixes to avoid collisions.

> For example, to avoid key collisions you may put your doctrine cache under different prefix than the rest of the
> application. Or if you have multiple applications using same memcached server, you could as long as prefixes are 
> unique you can avoid all key collisions.

# Configuration
```yaml
# Default configuration for extension with alias: "cyber_cache"
cyber_cache:
    # Select which engine to use for storage.
    engine:               memcache # One of "memcache"; "redis"
    host:                 ~ # Required
    port:                 ~

    # Global prefix
    prefix:               ~ # Required

    # Array of prefixes. A service will be generated for each element in this array to access the particular prefix.
    sub_prefixes:         []
```

## Services
The services provided by this bundle extend the native classes of corresponding **engine** (ex. `\Memcache`, `\Redis`), 
so you can replace all your memcache usage with these services directly.

The default service `Cyber\CacheBundle\Engine\CacheInterface` uses the global prefix, and can be injected using 
auto-wiring.

There will be an additional service created for each of the **sub_prefixes**. The service name of these 
services are in format `cyber.cache.%s` where **%s** is the value of sub-prefix.

> The full prefix of sub-prefixed services is `[global prefix]_[sub_prefix]`

## Usage
Use the services provided by this bundle like you would a regular `\Memcache` or `\Redis`.

First inject it into the class that needs `\Memcache` or `\Redis` access:
```yaml
services:
    App\MyServices:
        arguments:
            - '@Cyber\CacheBundle\Engine\CacheInterface' 
            # or any of your sub prefixed services: '@cyber.cache.example' (if your sub-prefix is 'example')
        
```

You can call the standard engine functions in your services (ex. `decrement` and `increment` for memcache engine, or `incr`
`decr` for redis engine). However these are there only to make these classes compatible with anything that expects 
native `\Memcache` or `\Redis` instances.

When writing your own code you should use our *unified* API which allows to easily switch between engines without
modifying your code. The api is defined inside `Cyber\CacheBundle\Engine\CacheInterface`. These are all the functions
starting with `c` such as `cGet()`. 

> Of course this api only provides a limited functionality as not all engines have same capabilities. But we try to
> implement most commonly used features there, so most of the time it will be sufficient. 

### Using native API 

```php
<?php

class MyServices
{
    // when you plan to use native API you should annotate your variable like so depending on your engine config,
    // this will tell your IDE to suggest native functions for autocompletion.
    /** @var \Redis */
    private $cache;
    
    public function __construct(CacheInterface $cache) {
        $this->cache = $cache;
        $this->cache->set('my_key', 'my_value');
    }
}
```

> Since `CacheInterface` implementations extends `\Memcache` or `\Redis` (depending on selected **engine**) it will 
> also fit into constructors like this `public function __construct(\Memcache $memcache)` or this 
> `public function __construct(\Redis $redis)` respectively

### Using Unified API

```php
<?php

class MyServices
{
    // when you plan to use unified API,
    // keep your annotation as CacheInterface to stop IDE from suggesting native functions
    /** @var CacheInterface */
    private $cache;
    
    public function __construct(CacheInterface $cache) {
        $this->cache = $cache;
        $this->cache->set('my_key', 'my_value');
    }
}
```

### Using Factory
Additionally you may inject `Cyber\CacheBundle\Engine\EngineFactory` to manually create  `CacheInterface` instances.

```php
<?php

class MyServices
{
    /** @var CacheInterface */
    private $cache;
    
    public function __construct(EngineFactory $factory) {
        $this->cache = $factory->create('my_prefix');
        $this->cache->set('my_key', 'my_value');
    }
}
```

# Session Handlers
A session handler implementation is provided for each of the caching engines. To use the session handler in your 
symfony application specify it in the **framework** configuration:

```yaml
framework:
    session:
        handler_id: Cyber\CacheBundle\Session\HandlerInterface
```

The interface will target the appropriate session handler based on the **engine** configuration of this bundle.
