<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CacheBundle\DependencyInjection;

use Cyber\CacheBundle\Engine\CacheInterface;
use Cyber\CacheBundle\Engine\Memcache\PrefixedMemcache;
use Cyber\CacheBundle\Engine\Redis\PrefixedRedis;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Loader;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

/**
 * This is the class that loads and manages your bundle configuration.
 *
 * To learn more see {@link http://symfony.com/doc/current/cookbook/bundles/extension.html}
 */
class CyberCacheExtension extends Extension
{
    const DEFAULT_PORTS = [
        'memcache'  => 11211,
        'redis'     => 6379,
    ];

    /**
     * {@inheritdoc}
     */
    public function load(array $configs, ContainerBuilder $container): void
    {
        $configuration = new Configuration();
        $config        = $this->processConfiguration($configuration, $configs);

        $loader = new Loader\YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/config'));
        $loader->load('services.yml');

        $container->setParameter('cyber.engine.host', $config['host']);
        $container->setParameter('cyber.engine.port', $config['port'] ?? self::DEFAULT_PORTS[$config['engine']]);
        $container->setParameter('cyber.engine.global_prefix', $config['prefix']);

        $this->makeRealEngine($config['engine'], $container);

        foreach ((array) $config['sub_prefixes'] as $prefix) {
            $this->makePrefixService($prefix, $container); //make one prefix for each sub service
        }
    }

    private function makePrefixService(string $prefix, ContainerBuilder $container): void
    {
        $container
            ->setDefinition(\sprintf('cyber.cache.%s', $prefix), new ChildDefinition('cyber.engine.abstract'))
            ->setArguments([$prefix])
            ->setPublic(true);
    }

    private function makeRealEngine(string $engine, ContainerBuilder $container): void
    {
        $realCacheEngine = $container->getDefinition('cyber.real.engine');
        $globalAlias     = $container->getDefinition(CacheInterface::class);

        switch ($engine) {
            case 'memcache':
                $realCacheEngine->setClass(\Memcache::class);
                $realCacheEngine->addMethodCall('addServer', ['%cyber.engine.host%', '%cyber.engine.port%']);
                $globalAlias->setClass(PrefixedMemcache::class);
                break;
            case 'redis':
                $realCacheEngine->setClass(\Redis::class);
                $globalAlias->addMethodCall('addServer', ['%cyber.engine.host%', '%cyber.engine.port%', 1]);
                $globalAlias->setClass(PrefixedRedis::class);
                break;
            default:
                throw new InvalidConfigurationException('Unknown engine: ' . $engine);
        }
    }
}
