<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CacheBundle\DependencyInjection;

use Cyber\CacheBundle\DependencyInjection\CyberCacheExtension;
use Cyber\CacheBundle\Engine\CacheInterface;
use Cyber\CacheBundle\Session\CyberCacheSessionHandler;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\DependencyInjection\ContainerBuilder;

class CyberCacheExtensionTest extends TestCase
{
    /**
     * @var CyberCacheExtension
     */
    private $extension;

    /**
     * Root name of the configuration.
     *
     * @var string
     */
    private $root;

    public function setUp(): void
    {
        parent::setUp();

        $this->extension = $this->getExtension();
        $this->root      = 'cyber_cache';
    }

    public function testGetConfigWithDefaultValues(): void
    {
        try {
            $this->extension->load([], $container = $this->getContainer());
        } catch (InvalidConfigurationException $ex) {
            $this->assertContains('host', $ex->getMessage());
        }

        try {
            $configs = [
                'host' => 'localhost',
            ];
            $this->extension->load([$configs], $container = $this->getContainer());
        } catch (InvalidConfigurationException $ex) {
            $this->assertContains('prefix', $ex->getMessage());
        }

        $configs = [
            'host'   => 'localhost',
            'prefix' => 'global',
        ];
        $this->extension->load([$configs], $container = $this->getContainer());

        $this->assertTrue($container->hasParameter('cyber.engine.host'));
        $this->assertEquals('localhost', $container->getParameter('cyber.engine.host'));
        $this->assertTrue($container->hasParameter('cyber.engine.port'));
        $this->assertEquals(CyberCacheExtension::DEFAULT_PORTS['memcache'], $container->getParameter('cyber.engine.port'));
        $this->assertTrue($container->hasParameter('cyber.engine.global_prefix'));
        $this->assertEquals('global', $container->getParameter('cyber.engine.global_prefix'));

        $this->assertTrue($container->hasDefinition('cyber.real.engine'));
        $this->assertEquals('Memcache', $container->getDefinition('cyber.real.engine')->getClass());
        $this->assertTrue($container->hasDefinition('cyber.engine.abstract'));
        $this->assertTrue($container->hasDefinition('Cyber\CacheBundle\Engine\EngineFactory'));
        $this->assertTrue($container->hasDefinition('Cyber\CacheBundle\Engine\CacheInterface'));

        $this->assertTrue($container->hasDefinition(CyberCacheSessionHandler::class));
    }

    public function testGetConfigWithOverrideValues(): void
    {
        /** @var array[] $configs */
        $configs = [
            'engine'       => 'redis',
            'host'         => 'cache.host',
            'port'         => 777,
            'prefix'       => 'local',
            'sub_prefixes' => [
                'sub',
            ],
        ];
        $this->extension->load([$configs], $container = $this->getContainer());

        $this->assertTrue($container->hasParameter('cyber.engine.host'));
        $this->assertEquals($configs['host'], $container->getParameter('cyber.engine.host'));
        $this->assertTrue($container->hasParameter('cyber.engine.port'));
        $this->assertEquals($configs['port'], $container->getParameter('cyber.engine.port'));
        $this->assertTrue($container->hasParameter('cyber.engine.global_prefix'));
        $this->assertEquals($configs['prefix'], $container->getParameter('cyber.engine.global_prefix'));

        $this->assertTrue($container->hasDefinition('cyber.real.engine'));
        $this->assertEquals('Redis', $container->getDefinition('cyber.real.engine')->getClass());
        $this->assertTrue($container->hasDefinition('cyber.engine.abstract'));
        $this->assertTrue($container->hasDefinition('Cyber\CacheBundle\Engine\EngineFactory'));
        $this->assertTrue($container->hasDefinition('Cyber\CacheBundle\Engine\CacheInterface'));
        foreach ($configs['sub_prefixes'] as $prefix) {
            $id = 'cyber.cache.' . $prefix;
            $this->assertTrue($container->hasDefinition($id));
            $this->assertEquals([$prefix], $container->getDefinition($id)->getArguments());
        }

        $this->assertTrue($container->hasDefinition(CyberCacheSessionHandler::class));
    }

    /**
     * @group integration
     */
    public function testCompiledContainer(): void
    {
        if (70200 > PHP_VERSION_ID) {
            $this->markTestSkipped('Requires php image 7.2');
        }

        $configs = [
            'engine'       => 'redis',
            'host'         => 'cache.host',
            'port'         => 777,
            'prefix'       => 'local',
            'sub_prefixes' => [
                'sub',
            ],
        ];
        $this->extension->load([$configs], $container = $this->getContainer());
        $container->compile();

        /** @var CacheInterface $cache */
        $cache = $container->get('cyber.cache.sub');
        $this->assertEquals('local_sub_', $cache->getFullPrefix());
    }

    /**
     * @group integration
     */
    public function testConfigurationDefaultPort(): void
    {
        $engineDefaultPorts = [
            'memcache'  => 11211,
            'redis'     => 6379,
        ];

        $configs = [
            'engine'       => 'engine.here',
            'host'         => 'cache.host',
            'prefix'       => 'local',
        ];

        // tests port defaults
        foreach ($engineDefaultPorts as $engine => $defaultPort) {
            $configs['engine'] = $engine;

            $this->extension->load([$configs], $container = $this->getContainer());
            $container->compile();

            $this->assertEquals($defaultPort, $container->getParameter('cyber.engine.port'));
        }

        // tests provided port
        $configs = [
            'engine'       => 'memcache',
            'host'         => 'cache.host',
            'prefix'       => 'local',
            'port'         => '8080',
        ];

        $this->extension->load([$configs], $container = $this->getContainer());
        $container->compile();

        $this->assertEquals($configs['port'], $container->getParameter('cyber.engine.port'));

        $configs = [
            'engine'       => 'redis',
            'host'         => 'cache.host',
            'prefix'       => 'local',
            'port'         => '12345',
        ];

        $this->extension->load([$configs], $container = $this->getContainer());
        $container->compile();

        $this->assertEquals($configs['port'], $container->getParameter('cyber.engine.port'));
    }

    /**
     * @return CyberCacheExtension
     */
    protected function getExtension(): CyberCacheExtension
    {
        return new CyberCacheExtension();
    }

    /**
     * @return ContainerBuilder
     */
    private function getContainer(): ContainerBuilder
    {
        $container = new ContainerBuilder();

        return $container;
    }
}
