<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CacheBundle\Engine\Memcache;

use Cyber\CacheBundle\Engine\Memcache\MemcacheList;
use Cyber\CacheBundle\Engine\Memcache\PrefixedMemcache;
use PHPUnit\Framework\TestCase;
use Tests\Cyber\CacheBundle\Mock\MockMemcache;

class MemcacheListTest extends TestCase
{
    /** @var MockMemcache */
    private $memcache;

    /** @var PrefixedMemcache */
    private $prefixedMemcache;

    public function testListSetOnEmptyList(): void
    {
        $list = new MemcacheList($this->prefixedMemcache, 'companies');
        $list->set(0, 'smith');
        $this->assertFalse($list->get(0)); // list is empty so set should fail
    }

    public function testConcurrentListCreation(): void
    {
        $list = new MemcacheList($this->prefixedMemcache, 'companies');

        $this->memcache->testsSetFailAdd(true);
        /** @var bool $result */
        $result = $list->push('should fail');

        $this->assertFalse($result);
    }

    public function testIntegration(): void
    {
        $list = new MemcacheList($this->prefixedMemcache, 'companies');

        $this->assertEquals(0, $list->getSize());
        /** @var bool $current */
        $current = $list->current();
        $this->assertFalse($current);
        /** @var bool $next */
        $next = $list->next();
        $this->assertFalse($next);
        /** @var bool $pop */
        $pop = $list->pop();
        $this->assertFalse($pop);

        $this->assertEquals(1, $list->push('smith'));
        $this->assertEquals(1, $list->getSize());
        $this->assertEquals(2, $list->push('joe'));
        $this->assertEquals(2, $list->getSize());
        $this->assertEquals('smith', $list->current());
        $this->assertEquals('smith', $list->next());
        $this->assertEquals('joe', $list->next());
        $this->assertEquals('smith', $list->next());
        $this->assertEquals('joe', $list->current());

        $this->assertEquals(3, $list->push('jane'));
        $this->assertEquals(3, $list->push($list->pop()));
        $this->assertEquals('joe', $list->pop());
        $this->assertEquals('jane', $list->current());

        $this->assertEquals(3, $list->push('zebra'));
        $this->assertEquals('jane', $list->get(0));
        $this->assertEquals('smith', $list->get(1));
        $this->assertEquals('zebra', $list->get(2));
        $this->assertEquals('zebra', $list->get(-1));

        $this->assertTrue($list->set(1, 'monkey'));
        $this->assertEquals('monkey', $list->get(1));
        $this->assertFalse($list->set(50, 'monkey'));
        $this->assertFalse($list->set(-50, 'monkey'));

        $this->assertTrue($list->clear());
        $this->assertEquals(0, $list->getSize());
        /** @var bool $next */
        $next = $list->next();
        $this->assertFalse($next);
    }

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        $this->memcache         = new MockMemcache();
        $this->prefixedMemcache = new PrefixedMemcache($this->memcache, 'test' . \round(\microtime(true), 3));
    }
}
