<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CacheBundle\Engine\Memcache;

use Cyber\CacheBundle\Engine\Memcache\MemcacheList;
use Cyber\CacheBundle\Engine\Memcache\PrefixedMemcache;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use Tests\Cyber\CacheBundle\Mock\MockMemcache;

class PrefixedMemcacheTest extends TestCase
{
    const PREFIX = 'rock';

    /** @var \Memcache|MockObject */
    private $memcache;

    /** @var PrefixedMemcache */
    private $pMemcache;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        $this->memcache  = $this->mockMemcache();
        $this->pMemcache = new PrefixedMemcache($this->memcache, self::PREFIX);
    }

    public function testAdd(): void
    {
        $expected = true;
        $value    = 'zebra';
        $flags    = 7;
        $expire   = 777;

        $this->memcache->expects($this->once())
            ->method('add')
            ->with(self::PREFIX . '_apple', $value, $flags, $expire)
            ->willReturn($expected);

        $result = $this->pMemcache->add('apple', $value, $flags, $expire);
        $this->assertEquals($expected, $result);
    }

    public function testGet(): void
    {
        $expected = 7;
        $flags    = 8;

        $this->memcache->expects($this->once())
            ->method('get')
            ->with(self::PREFIX . '_apple', $flags)
            ->willReturn($expected);

        $result = $this->pMemcache->get('apple', $flags);
        $this->assertEquals($expected, $result);
    }

    public function testGetList(): void
    {
        $result = $this->pMemcache->getList('test');

        $this->assertInstanceOf(MemcacheList::class, $result);
        $refl = new ReflectionClass($result);
        $prop = $refl->getProperty('id');
        $prop->setAccessible(true);
        $id = $prop->getValue($result);
        $this->assertEquals('cyber_list[test]', $id);

        $result2 = $this->pMemcache->getList('test');
        $this->assertSame($result, $result2); // should return same object on second call to same key
    }

    /**
     * @depends testGetList
     */
    public function testSetList(): void
    {
        $mockList = $this->getMockList();
        $this->pMemcache->setList('manual', $mockList);
        $this->assertSame($mockList, $this->pMemcache->getList('manual'));
    }

    /**
     * @depends      testSetList
     * @dataProvider listOperationsProvier
     *
     * @param string $func
     * @param mixed  $arg
     * @param mixed  $listFunc
     * @param mixed  $listArg
     */
    public function testListOperations($func, $arg, $listFunc, $listArg): void
    {
        $expected = \in_array($func, ['cListSet', 'cListClear'], true) ?: 'smith';
        $mockList = $this->getMockList();
        $mockList->expects($this->once())
            ->method($listFunc)
            ->withConsecutive($listArg)
            ->willReturn($expected);
        $this->pMemcache->setList('manual', $mockList);

        \array_unshift($arg, 'manual');

        $callable = [$this->pMemcache, $func];
        $this->assertTrue(\is_object($this->pMemcache) && \is_callable($callable));

        $result = \call_user_func_array($callable, $arg);
        $this->assertEquals($expected, $result);
    }

    /**
     * @return array<array<mixed>>
     */
    public function listOperationsProvier(): array
    {
        return [
            'cListGet'         => ['cListGet', ['1'], 'get', ['1']],
            'cListPop'         => ['cListPop', [], 'pop', []],
            'cListPush'        => ['cListPush', ['x'], 'push', ['x']],
            'cListSet'         => ['cListSet', ['1', 'x'], 'set', ['1', 'x']],
            'cListCircularGet' => ['cListCircularGet', [], 'next', []],
            'cListClear'       => ['cListClear', [], 'clear', []],
        ];
    }

    /**
     * @return \Memcache|MockObject
     */
    private function mockMemcache()
    {
        return $this->getMockBuilder(MockMemcache::class)
            ->disableOriginalConstructor()
            ->getMock();
    }

    /**
     * @return MemcacheList|MockObject
     */
    private function getMockList()
    {
        return $this->getMockBuilder(MemcacheList::class)
            ->disableOriginalConstructor()
            ->getMock();
    }
}
