<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CacheBundle\Engine;

interface CacheInterface
{
    /**
     * @param mixed  $engine
     * @param string $prefix
     */
    public function __construct($engine, string $prefix);

    public function getPrefix(): string;

    /**
     * If this is the top level object, this function will return same value as getPrefix().
     *
     * If this object has another PrefixedMemcache as parent, the combined prefix of all parents will get returned.
     *
     * @return string prefix of this and all parent objects
     */
    public function getFullPrefix(): string;

    /**
     * @param string|string[] $key
     *
     * @return string|string[] the key (or keys if arg was array) prepended with configured prefix
     */
    public function addPrefix($key);

    /**
     * Add a key if it does not exist.
     *
     * @param string $key
     * @param string $value
     * @param int    $ttl   time to live in seconds
     *
     * @return bool true on success false on failure
     */
    public function cAdd($key, $value, $ttl = 0): bool;

    /**
     * Set a value for a key.
     *
     * @param string $key
     * @param string $value
     * @param int    $ttl   time to live in seconds
     *
     * @return bool true on success false on failure
     */
    public function cSet($key, $value, $ttl = 0): bool;

    /**
     * Replace a key with new value only if key exists.
     *
     * Same as set but will fail if key does not exist
     *
     * @param string $key
     * @param string $value
     * @param int    $ttl
     *
     * @return bool
     */
    public function cReplace($key, $value, $ttl = 0): bool;

    /**
     * Get a value of a key.
     *
     * @param string $key
     *
     * @return false|string value of key or FALSE on failure
     */
    public function cGet($key);

    /**
     * Deletes they key.
     *
     * @param string $key
     *
     * @return bool
     */
    public function cDelete($key): bool;

    /**
     * Increments the key by value.
     *
     * Will create a key if one does not exist
     *
     * @param string $key
     * @param int    $value
     *
     * @return bool|int
     */
    public function cIncr($key, $value = 1);

    /**
     * Decrements the key by value.
     *
     * Will create a key if one does not exist
     *
     * @param string $key
     * @param int    $value
     *
     * @return bool|int
     */
    public function cDecr($key, $value = 1);

    /**
     * Removes all data from database.
     *
     * @return bool
     */
    public function cFlush(): bool;

    /**
     * Set the string value of a key and return its old value.
     *
     * @param string $key   key to replace
     * @param string $value value to replace with
     *
     * @return bool|string previous value of the key
     */
    public function cGetSet($key, $value);

    /**
     * @param string $key   list key
     * @param int    $index list index
     *
     * @return mixed value at the specified list index
     */
    public function cListGet($key, $index);

    /**
     * @param string $key list key
     *
     * @return bool|string the first value in the list and remove it from the list
     */
    public function cListPop($key);

    /**
     * Pushes a new value to the end of the list.
     *
     * @param string $key   list key
     * @param string $value
     *
     * @return bool|int size of the list or FALSE on failure
     */
    public function cListPush($key, $value);

    /**
     * @param int    $index index of the value to set
     * @param string $value value to set at this index
     * @param string $key   list key
     *
     * @return bool true on success false on failure, or if index it out of bounds of the list
     */
    public function cListSet($key, $index, $value): bool;

    /**
     * Moves internal pointer to the next item in the list.
     *
     * Can be used to provide round-robin mechanism:
     * 1. load all values to use for round-robin using cListPush()
     * 2. call cListRotateGet() to get the next item in the list,
     *
     * When end of the list is reached pointer goes back to the beginning.
     *
     * @param string $key list key
     *
     * @return bool|string data being pointed by internal pointer before advancing it to the next; FALSE on empty list or error
     */
    public function cListCircularGet($key);

    /**
     * Clears all list data under specified $key.
     *
     * @param string $key list key
     *
     * @return bool true if operation succeeds
     */
    public function cListClear($key): bool;
}
