<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CacheBundle\Engine\Redis;

use Cyber\CacheBundle\Engine\CacheInterface;
use Cyber\CacheBundle\Engine\EngineTrait;
use InvalidArgumentException;
use Redis;

/**
 * Class PrefixedRedis.
 *
 * @SuppressWarnings(PHPMD.CamelCaseParameterName)
 * @SuppressWarnings(PHPMD.ExcessiveClassLength)
 * @SuppressWarnings(PHPMD.TooManyMethods)
 * @SuppressWarnings(PHPMD.TooManyPublicMethods)
 * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
 * @SuppressWarnings(PHPMD.ExcessivePublicCount)
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
class PrefixedRedis extends Redis implements CacheInterface
{
    use EngineTrait;

    /** @var PrefixedRedis|Redis */
    protected $engine;

    /** @var array<mixed> */
    private $serverArgs;

    /** @var bool */
    private $connected = false;

    /**
     * @param PrefixedRedis&Redis $engine
     * @param string              $prefix
     *
     * @throws \RuntimeException when memcache extension is not enabled
     */
    public function __construct($engine, string $prefix)
    {
        if (!$engine instanceof Redis) {
            throw new InvalidArgumentException('Engine must be instance of \Memcache');
        }

        parent::__construct();

        $this->engine = $engine;
        $this->prefix = $prefix;
    }

    public function addServer(
        $host,
        $port = null,
        $timeout = 1,
        $reserved = null,
        $retry_interval = null,
        $read_timeout = 0
    ) {
        $this->serverArgs = \func_get_args();
    }

    /**
     * {@inheritdoc}
     */
    public function connect(
        $host,
        $port = null,
        $timeout = 1,
        $reserved = null,
        $retry_interval = null,
        $read_timeout = 0
    ) {
        \trigger_error('Connection is managed by the factory.', E_USER_WARNING);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function open($host, $port = 6379, $timeout = 0.0, $retry_interval = 0)
    {
        \trigger_error('Connection is managed by the factory.', E_USER_WARNING);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function pconnect($host, $port = null, $timeout = 0.0, $persistent_id = null)
    {
        \trigger_error('Connection is managed by the factory.', E_USER_WARNING);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function popen($host, $port = 6379, $timeout = 0.0, $persistent_id = null)
    {
        \trigger_error('Connection is managed by the factory.', E_USER_WARNING);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function close()
    {
        \trigger_error('Closing Cache Bundle Engine is not allowed. Connection managed by factory.', E_USER_WARNING);

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function setOption($name, $value)
    {
        $this->doConnect();

        return $this->engine->setOption($name, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function getOption($name)
    {
        $this->doConnect();

        return $this->engine->getOption($name);
    }

    /**
     * {@inheritdoc}
     */
    public function ping()
    {
        $this->doConnect();

        return $this->engine->ping();
    }

    /**
     * {@inheritdoc}
     */
    public function setex($key, $ttl, $value)
    {
        $this->doConnect();

        return $this->engine->setex($this->addPrefix($key), $ttl, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function psetex($key, $ttl, $value)
    {
        $this->doConnect();

        return $this->engine->psetex($this->addPrefix($key), $ttl, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function sScan($key, &$iterator, $pattern = null, $count = null)
    {
        $this->doConnect();

        return $this->engine->sScan($this->addPrefix($key), $iterator, $pattern, $count);
    }

    /**
     * {@inheritdoc}
     */
    public function scan(&$iterator, $pattern = null, $count = null)
    {
        $this->doConnect();

        return $this->engine->scan($iterator, $pattern, $count);
    }

    /**
     * {@inheritdoc}
     */
    public function zScan($key, &$iterator, $pattern = null, $count = null)
    {
        $this->doConnect();

        return $this->engine->zScan($this->addPrefix($key), $iterator, $pattern, $count);
    }

    /**
     * {@inheritdoc}
     */
    public function hScan($key, &$iterator, $pattern = null, $count = null)
    {
        $this->doConnect();

        return $this->engine->hScan($this->addPrefix($key), $iterator, $pattern, $count);
    }

    /**
     * {@inheritdoc}
     */
    public function client($command, ...$args)
    {
        $this->doConnect();

        return $this->engine->client($command, ...$args);
    }

    /**
     * {@inheritdoc}
     */
    public function slowlog($command, $option = null)
    {
        $this->doConnect();

        return $this->engine->slowlog($command, $option);
    }

    /**
     * {@inheritdoc}
     */
    public function get($key)
    {
        $this->doConnect();

        return $this->engine->get($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function set($key, $value, $timeout = null)
    {
        $this->doConnect();

        return $this->engine->set($this->addPrefix($key), $value, $timeout);
    }

    /**
     * {@inheritdoc}
     */
    public function setnx($key, $value)
    {
        $this->doConnect();

        return $this->engine->setnx($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function del($key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->del($this->addPrefix($key1), ...$this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function delete($key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->delete($this->addPrefix($key1), ...$this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function multi($mode = Redis::MULTI)
    {
        $this->doConnect();

        return $this->engine->multi($mode);
    }

    /**
     * {@inheritdoc}
     */
    public function exec()
    {
        $this->doConnect();

        return $this->engine->exec();
    }

    /**
     * {@inheritdoc}
     */
    public function discard()
    {
        $this->doConnect();

        return $this->engine->discard();
    }

    /**
     * {@inheritdoc}
     */
    public function watch($key, ...$otherKeys)
    {
        $this->doConnect();

        $this->engine->watch($this->addPrefix($key), $this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function unwatch()
    {
        $this->doConnect();

        return $this->engine->unwatch();
    }

    /**
     * {@inheritdoc}
     */
    public function subscribe($channels, $callback = null)
    {
        $this->doConnect();

        return $this->engine->subscribe($channels, $callback);
    }

    /**
     * {@inheritdoc}
     */
    public function psubscribe($patterns, $callback = null)
    {
        $this->doConnect();

        return $this->engine->psubscribe($patterns, $callback);
    }

    /**
     * {@inheritdoc}
     */
    public function publish($channel, $message)
    {
        $this->doConnect();

        return $this->engine->publish($channel, $message);
    }

    /**
     * {@inheritdoc}
     */
    public function pubsub($keyword, ...$argument)
    {
        $this->doConnect();

        return $this->engine->pubsub($keyword, $argument);
    }

    /**
     * {@inheritdoc}
     */
    public function exists($key, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->exists($this->addPrefix($key), $this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function incr($key)
    {
        $this->doConnect();

        return $this->engine->incr($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function incrByFloat($key, $increment)
    {
        $this->doConnect();

        return $this->engine->incrByFloat($this->addPrefix($key), $increment);
    }

    /**
     * {@inheritdoc}
     */
    public function incrBy($key, $value)
    {
        $this->doConnect();

        return $this->engine->incrBy($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function decr($key)
    {
        $this->doConnect();

        return $this->engine->decr($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function decrBy($key, $value)
    {
        $this->doConnect();

        return $this->engine->decrBy($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function getMultiple(array $keys)
    {
        $this->doConnect();

        return $this->engine->getMultiple((array) $this->addPrefix($keys));
    }

    /**
     * {@inheritdoc}
     */
    public function lPush($key, $value1, $value2 = null, $valueN = null)
    {
        $this->doConnect();

        return $this->engine->lPush($this->addPrefix($key), $value1, $value2, $valueN);
    }

    /**
     * {@inheritdoc}
     */
    public function rPush($key, $value1, $value2 = null, $valueN = null)
    {
        $this->doConnect();

        return $this->engine->rPush($this->addPrefix($key), $value1, $value2, $valueN);
    }

    /**
     * {@inheritdoc}
     */
    public function lPushx($key, $value)
    {
        $this->doConnect();

        return $this->engine->lPushx($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function rPushx($key, $value)
    {
        $this->doConnect();

        return $this->engine->rPushx($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function lPop($key)
    {
        $this->doConnect();

        return $this->engine->lPop($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function rPop($key)
    {
        $this->doConnect();

        return $this->engine->rPop($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function blPop($key, $timeoutOrKey, ...$extraArgs)
    {
        $this->doConnect();

        $data    = \array_merge([$key, $timeoutOrKey], $extraArgs);
        $timeout = \array_pop($data);

        return $this->engine->blPop($this->addPrefix($data), $timeout);
    }

    /**
     * {@inheritdoc}
     */
    public function brPop($key, $timeoutOrKey, ...$extraArgs)
    {
        $this->doConnect();

        $data    = \array_merge([$key, $timeoutOrKey], $extraArgs);
        $timeout = \array_pop($data);

        return $this->engine->brPop($this->addPrefix($data), $timeout);
    }

    /**
     * {@inheritdoc}
     */
    public function lLen($key)
    {
        $this->doConnect();

        return $this->engine->lLen($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function lSize($key)
    {
        $this->doConnect();

        return $this->engine->lSize($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function lIndex($key, $index)
    {
        $this->doConnect();

        return $this->engine->lIndex($this->addPrefix($key), $index);
    }

    /**
     * {@inheritdoc}
     */
    public function lGet($key, $index)
    {
        $this->doConnect();

        return $this->engine->lGet($this->addPrefix($key), $index);
    }

    /**
     * {@inheritdoc}
     */
    public function lSet($key, $index, $value)
    {
        $this->doConnect();

        return $this->engine->lSet($this->addPrefix($key), $index, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function lRange($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->lRange($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function lGetRange($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->lGetRange($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function lTrim($key, $start, $stop)
    {
        $this->doConnect();

        return $this->engine->lTrim($this->addPrefix($key), $start, $stop);
    }

    /**
     * {@inheritdoc}
     */
    public function listTrim($key, $start, $stop)
    {
        $this->doConnect();

        return $this->engine->listTrim($this->addPrefix($key), $start, $stop);
    }

    /**
     * {@inheritdoc}
     */
    public function lRem($key, $value, $count)
    {
        $this->doConnect();

        return $this->engine->lRem($this->addPrefix($key), $value, $count);
    }

    /**
     * {@inheritdoc}
     */
    public function lRemove($key, $value, $count)
    {
        $this->doConnect();

        return $this->engine->lRemove($this->addPrefix($key), $value, $count);
    }

    /**
     * {@inheritdoc}
     */
    public function lInsert($key, $position, $pivot, $value)
    {
        $this->doConnect();

        return $this->engine->lInsert($this->addPrefix($key), $position, $pivot, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function sAdd($key, $value1, $value2 = null, $valueN = null)
    {
        $this->doConnect();

        return $this->engine->sAdd($this->addPrefix($key), $value1, $value2, $valueN);
    }

    /**
     * {@inheritdoc}
     */
    public function sAddArray($key, array $values)
    {
        $this->doConnect();

        return $this->engine->sAddArray($this->addPrefix($key), $values);
    }

    /**
     * {@inheritdoc}
     */
    public function sRem($key, $member1, ...$otherMembers)
    {
        $this->doConnect();

        return $this->engine->sRem($this->addPrefix($key), $member1, ...$otherMembers);
    }

    /**
     * {@inheritdoc}
     */
    public function sRemove($key, $member1, ...$otherMembers)
    {
        $this->doConnect();

        return $this->engine->sRemove($this->addPrefix($key), $member1, ...$otherMembers);
    }

    /**
     * {@inheritdoc}
     */
    public function sMove($srcKey, $dstKey, $member)
    {
        $this->doConnect();

        return $this->engine->sMove($this->addPrefix($srcKey), $this->addPrefix($dstKey), $member);
    }

    /**
     * {@inheritdoc}
     */
    public function sIsMember($key, $value)
    {
        $this->doConnect();

        return $this->engine->sIsMember($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function sContains($key, $value)
    {
        $this->doConnect();

        return $this->engine->sContains($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function sCard($key)
    {
        $this->doConnect();

        return $this->engine->sCard($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function sPop($key)
    {
        $this->doConnect();

        return $this->engine->sPop($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function sRandMember($key, $count = null)
    {
        $this->doConnect();

        return $this->engine->sRandMember($this->addPrefix($key), $count);
    }

    /**
     * {@inheritdoc}
     */
    public function sInter($key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->sInter($this->addPrefix($key1), ...$this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function sInterStore($dstKey, $key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->sInterStore(
            $this->addPrefix($dstKey),
            $this->addPrefix($key1),
            ...$this->addPrefix($otherKeys)
        );
    }

    /**
     * {@inheritdoc}
     */
    public function sUnion($key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->sUnion($this->addPrefix($key1), ...$this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function sUnionStore($dstKey, $key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->sUnionStore(
            $this->addPrefix($dstKey),
            $this->addPrefix($key1),
            ...$this->addPrefix($otherKeys)
        );
    }

    /**
     * {@inheritdoc}
     */
    public function sDiff($key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->sDiff($this->addPrefix($key1), ...$this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function sDiffStore($dstKey, $key1, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->sDiffStore(
            $this->addPrefix($dstKey),
            $this->addPrefix($key1),
            ...$this->addPrefix($otherKeys)
        );
    }

    /**
     * {@inheritdoc}
     */
    public function sMembers($key)
    {
        $this->doConnect();

        return $this->engine->sMembers($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function sGetMembers($key)
    {
        $this->doConnect();

        return $this->engine->sGetMembers($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function getSet($key, $value)
    {
        $this->doConnect();

        return $this->engine->getSet($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function randomKey()
    {
        $this->doConnect();

        return $this->engine->randomKey();
    }

    /**
     * {@inheritdoc}
     */
    public function select($dbindex)
    {
        $this->doConnect();

        return $this->engine->select($dbindex);
    }

    /**
     * {@inheritdoc}
     */
    public function move($key, $dbindex)
    {
        $this->doConnect();

        return $this->engine->move($this->addPrefix($key), $dbindex);
    }

    /**
     * {@inheritdoc}
     */
    public function rename($srcKey, $dstKey)
    {
        $this->doConnect();

        return $this->engine->rename($this->addPrefix($srcKey), $this->addPrefix($dstKey));
    }

    /**
     * {@inheritdoc}
     */
    public function renameKey($srcKey, $dstKey)
    {
        $this->doConnect();

        return $this->engine->renameKey($this->addPrefix($srcKey), $this->addPrefix($dstKey));
    }

    /**
     * {@inheritdoc}
     */
    public function renameNx($srcKey, $dstKey)
    {
        $this->doConnect();

        return $this->engine->renameNx($this->addPrefix($srcKey), $this->addPrefix($dstKey));
    }

    /**
     * {@inheritdoc}
     */
    public function expire($key, $ttl)
    {
        $this->doConnect();

        return $this->engine->expire($this->addPrefix($key), $ttl);
    }

    /**
     * {@inheritdoc}
     */
    public function pExpire($key, $ttl)
    {
        $this->doConnect();

        return $this->engine->pExpire($this->addPrefix($key), $ttl);
    }

    /**
     * {@inheritdoc}
     */
    public function setTimeout($key, $ttl)
    {
        $this->doConnect();

        return $this->engine->setTimeout($this->addPrefix($key), $ttl);
    }

    /**
     * {@inheritdoc}
     */
    public function expireAt($key, $timestamp)
    {
        $this->doConnect();

        return $this->engine->expireAt($this->addPrefix($key), $timestamp);
    }

    /**
     * {@inheritdoc}
     */
    public function pExpireAt($key, $timestamp)
    {
        $this->doConnect();

        return $this->engine->pExpireAt($this->addPrefix($key), $timestamp);
    }

    /**
     * {@inheritdoc}
     */
    public function keys($pattern)
    {
        $this->doConnect();

        return $this->engine->keys($pattern);
    }

    /**
     * {@inheritdoc}
     */
    public function getKeys($pattern)
    {
        $this->doConnect();

        return $this->engine->getKeys($pattern);
    }

    /**
     * {@inheritdoc}
     */
    public function dbSize()
    {
        $this->doConnect();

        return $this->engine->dbSize();
    }

    /**
     * {@inheritdoc}
     */
    public function auth($password)
    {
        $this->doConnect();

        return $this->engine->auth($password);
    }

    /**
     * {@inheritdoc}
     */
    public function bgrewriteaof()
    {
        $this->doConnect();

        return $this->engine->bgrewriteaof();
    }

    /**
     * {@inheritdoc}
     */
    public function slaveof($host = '127.0.0.1', $port = 6379)
    {
        $this->doConnect();

        return $this->engine->slaveof($host, $port);
    }

    /**
     * {@inheritdoc}
     */
    public function object($string = '', $key = '')
    {
        $this->doConnect();

        return $this->engine->object($string, $this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function save()
    {
        $this->doConnect();

        return $this->engine->save();
    }

    /**
     * {@inheritdoc}
     */
    public function bgsave()
    {
        $this->doConnect();

        return $this->engine->bgsave();
    }

    /**
     * {@inheritdoc}
     */
    public function lastSave()
    {
        $this->doConnect();

        return $this->engine->lastSave();
    }

    /**
     * {@inheritdoc}
     */
    public function wait($numSlaves, $timeout)
    {
        $this->doConnect();

        return $this->engine->wait($numSlaves, $timeout);
    }

    /**
     * {@inheritdoc}
     */
    public function type($key)
    {
        $this->doConnect();

        return $this->engine->type($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function append($key, $value)
    {
        $this->doConnect();

        return $this->engine->append($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function getRange($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->getRange($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function substr($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->substr($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function setRange($key, $offset, $value)
    {
        $this->doConnect();

        return $this->engine->setRange($this->addPrefix($key), $offset, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function strlen($key)
    {
        $this->doConnect();

        return $this->engine->strlen($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function bitpos($key, $bit, $start = 0, $end = null)
    {
        $this->doConnect();

        return $this->engine->bitpos($this->addPrefix($key), $bit, $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function getBit($key, $offset)
    {
        $this->doConnect();

        return $this->engine->getBit($this->addPrefix($key), $offset);
    }

    /**
     * {@inheritdoc}
     */
    public function setBit($key, $offset, $value)
    {
        $this->doConnect();

        return $this->engine->setBit($this->addPrefix($key), $offset, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function bitCount($key)
    {
        $this->doConnect();

        return $this->engine->bitCount($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function bitOp($operation, $retKey, $key, ...$otherKeys)
    {
        $this->doConnect();

        return $this->engine->bitOp($operation, $retKey, $this->addPrefix($key), $this->addPrefix($otherKeys));
    }

    /**
     * {@inheritdoc}
     */
    public function flushDB($async = null)
    {
        $this->doConnect();

        return $this->engine->flushDB($async);
    }

    /**
     * {@inheritdoc}
     */
    public function flushAll($async = null)
    {
        $this->doConnect();

        return $this->engine->flushAll($async);
    }

    /**
     * {@inheritdoc}
     */
    public function sort($key, $option = null)
    {
        $this->doConnect();

        return $this->engine->sort($this->addPrefix($key), $option);
    }

    /**
     * {@inheritdoc}
     */
    public function info($option = null)
    {
        $this->doConnect();

        return $this->engine->info($option);
    }

    /**
     * {@inheritdoc}
     */
    public function resetStat()
    {
        $this->doConnect();

        return $this->engine->resetStat();
    }

    /**
     * {@inheritdoc}
     */
    public function ttl($key)
    {
        $this->doConnect();

        return $this->engine->ttl($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function pttl($key)
    {
        $this->doConnect();

        return $this->engine->pttl($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function persist($key)
    {
        $this->doConnect();

        return $this->engine->persist($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function mset(array $array)
    {
        $this->doConnect();

        return $this->engine->mset($array);
    }

    /**
     * {@inheritdoc}
     */
    public function mget(array $array)
    {
        $this->doConnect();

        return $this->engine->mget($array);
    }

    /**
     * {@inheritdoc}
     */
    public function msetnx(array $array)
    {
        $this->doConnect();

        return $this->engine->msetnx($array);
    }

    /**
     * {@inheritdoc}
     */
    public function rpoplpush($srcKey, $dstKey)
    {
        $this->doConnect();

        return $this->engine->rpoplpush($this->addPrefix($srcKey), $this->addPrefix($dstKey));
    }

    /**
     * {@inheritdoc}
     */
    public function brpoplpush($srcKey, $dstKey, $timeout)
    {
        $this->doConnect();

        return $this->engine->brpoplpush($this->addPrefix($srcKey), $this->addPrefix($dstKey), $timeout);
    }

    /**
     * {@inheritdoc}
     */
    public function zAdd($key, $score1, $value1, ...$extraArgs)
    {
        $this->doConnect();

        return $this->engine->zAdd($this->addPrefix($key), $score1, $value1, ...$extraArgs);
    }

    /**
     * {@inheritdoc}
     */
    public function zRange($key, $start, $end, $withscores = null)
    {
        $this->doConnect();

        return $this->engine->zRange($this->addPrefix($key), $start, $end, $withscores);
    }

    /**
     * {@inheritdoc}
     */
    public function zRem($key, $member1, ...$otherMembers)
    {
        $this->doConnect();

        return $this->engine->zRem($this->addPrefix($key), $member1, ...$otherMembers);
    }

    /**
     * {@inheritdoc}
     */
    public function zDelete($key, $member1, ...$otherMembers)
    {
        $this->doConnect();

        return $this->engine->zDelete($this->addPrefix($key), $member1, ...$otherMembers);
    }

    /**
     * {@inheritdoc}
     */
    public function zRevRange($key, $start, $end, $withscore = null)
    {
        $this->doConnect();

        return $this->engine->zRevRange($this->addPrefix($key), $start, $end, $withscore);
    }

    /**
     * {@inheritdoc}
     */
    public function zRangeByScore($key, $start, $end, array $options = [])
    {
        $this->doConnect();

        return $this->engine->zRangeByScore($this->addPrefix($key), $start, $end, $options);
    }

    /**
     * {@inheritdoc}
     */
    public function zRevRangeByScore($key, $start, $end, array $options = [])
    {
        $this->doConnect();

        return $this->engine->zRevRangeByScore($this->addPrefix($key), $start, $end, $options);
    }

    /**
     * {@inheritdoc}
     */
    public function zRangeByLex($key, $min, $max, $offset = null, $limit = null)
    {
        $this->doConnect();

        return $this->engine->zRangeByLex($this->addPrefix($key), $min, $max, $offset, $limit);
    }

    /**
     * {@inheritdoc}
     */
    public function zRevRangeByLex($key, $min, $max, $offset = null, $limit = null)
    {
        $this->doConnect();

        return $this->engine->zRevRangeByLex($this->addPrefix($key), $min, $max, $offset, $limit);
    }

    /**
     * {@inheritdoc}
     */
    public function zCount($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->zCount($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function zRemRangeByScore($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->zRemRangeByScore($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function zDeleteRangeByScore($key, $start, $end)
    {
        $this->doConnect();
        $this->engine->zDeleteRangeByScore($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function zRemRangeByRank($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->zRemRangeByRank($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function zDeleteRangeByRank($key, $start, $end)
    {
        $this->doConnect();

        return $this->engine->zDeleteRangeByRank($this->addPrefix($key), $start, $end);
    }

    /**
     * {@inheritdoc}
     */
    public function zCard($key)
    {
        $this->doConnect();

        return $this->engine->zCard($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function zSize($key)
    {
        $this->doConnect();

        return $this->engine->zSize($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function zScore($key, $member)
    {
        $this->doConnect();

        return $this->engine->zScore($this->addPrefix($key), $member);
    }

    /**
     * {@inheritdoc}
     */
    public function zRank($key, $member)
    {
        $this->doConnect();

        return $this->engine->zRank($this->addPrefix($key), $member);
    }

    /**
     * {@inheritdoc}
     */
    public function zRevRank($key, $member)
    {
        $this->doConnect();

        return $this->engine->zRevRank($this->addPrefix($key), $member);
    }

    /**
     * {@inheritdoc}
     */
    public function zIncrBy($key, $value, $member)
    {
        $this->doConnect();

        return $this->engine->zIncrBy($this->addPrefix($key), $value, $member);
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.CamelCaseParameterName)
     * @SuppressWarnings(PHPMD.CamelCaseVariableName)
     */
    public function zUnion($Output, $ZSetKeys, array $Weights = null, $aggregateFunction = 'SUM')
    {
        $this->doConnect();

        return $this->engine->zUnion($Output, $this->addPrefix($ZSetKeys), $Weights, $aggregateFunction);
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.CamelCaseParameterName)
     * @SuppressWarnings(PHPMD.CamelCaseVariableName)
     */
    public function zInter($Output, $ZSetKeys, array $Weights = null, $aggregateFunction = 'SUM')
    {
        $this->doConnect();

        return $this->engine->zInter($Output, $this->addPrefix($ZSetKeys), $Weights, $aggregateFunction);
    }

    /**
     * {@inheritdoc}
     */
    public function hSet($key, $hashKey, $value)
    {
        $this->doConnect();

        return $this->engine->hSet($this->addPrefix($key), $hashKey, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function hSetNx($key, $hashKey, $value)
    {
        $this->doConnect();

        return $this->engine->hSetNx($this->addPrefix($key), $hashKey, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function hGet($key, $hashKey)
    {
        $this->doConnect();

        return $this->engine->hGet($this->addPrefix($key), $hashKey);
    }

    /**
     * {@inheritdoc}
     */
    public function hLen($key)
    {
        $this->doConnect();

        return $this->engine->hLen($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function hDel($key, $hashKey1, ...$otherMembers)
    {
        $this->doConnect();

        return $this->engine->hDel($this->addPrefix($key), $hashKey1, ...$otherMembers);
    }

    /**
     * {@inheritdoc}
     */
    public function hKeys($key)
    {
        $this->doConnect();

        return $this->engine->hKeys($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function hVals($key)
    {
        $this->doConnect();

        return $this->engine->hVals($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function hGetAll($key)
    {
        $this->doConnect();

        return $this->engine->hGetAll($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function hExists($key, $hashKey)
    {
        $this->doConnect();

        return $this->engine->hExists($this->addPrefix($key), $hashKey);
    }

    /**
     * {@inheritdoc}
     */
    public function hIncrBy($key, $hashKey, $value)
    {
        $this->doConnect();

        return $this->engine->hIncrBy($this->addPrefix($key), $hashKey, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function hIncrByFloat($key, $field, $increment)
    {
        $this->doConnect();

        return $this->engine->hIncrByFloat($this->addPrefix($key), $field, $increment);
    }

    /**
     * {@inheritdoc}
     */
    public function hMset($key, $hashKeys)
    {
        $this->doConnect();

        return $this->engine->hMset($this->addPrefix($key), $hashKeys);
    }

    /**
     * {@inheritdoc}
     */
    public function hMGet($key, $hashKeys)
    {
        $this->doConnect();

        return $this->engine->hMGet($this->addPrefix($key), $hashKeys);
    }

    /**
     * {@inheritdoc}
     */
    public function config($operation, $key, $value = null)
    {
        $this->doConnect();

        return $this->engine->config($operation, $this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function evaluate($script, $args = [], $numKeys = 0)
    {
        $this->doConnect();

        return $this->engine->evaluate($script, $args, $numKeys);
    }

    /**
     * {@inheritdoc}
     */
    public function evalSha($scriptSha, $args = [], $numKeys = 0)
    {
        $this->doConnect();

        return $this->engine->evalSha($scriptSha, $args, $numKeys);
    }

    /**
     * {@inheritdoc}
     */
    public function evaluateSha($scriptSha, $args = [], $numKeys = 0)
    {
        $this->doConnect();

        return $this->engine->evaluateSha($scriptSha, $args, $numKeys);
    }

    /**
     * {@inheritdoc}
     */
    public function script($command, ...$script)
    {
        $this->doConnect();

        return $this->engine->script($command, ...$script);
    }

    /**
     * {@inheritdoc}
     */
    public function getLastError()
    {
        $this->doConnect();

        return $this->engine->getLastError();
    }

    /**
     * {@inheritdoc}
     */
    public function clearLastError()
    {
        $this->doConnect();

        return $this->engine->clearLastError();
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.CamelCaseMethodName)
     */
    public function _prefix($value)
    {
        $this->doConnect();

        return $this->engine->_prefix($value);
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.CamelCaseMethodName)
     */
    public function _unserialize($value)
    {
        $this->doConnect();

        return $this->engine->_unserialize($value);
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.CamelCaseMethodName)
     */
    public function _serialize($value)
    {
        $this->doConnect();

        return $this->engine->_serialize($value);
    }

    /**
     * {@inheritdoc}
     */
    public function dump($key)
    {
        $this->doConnect();

        return $this->engine->dump($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function restore($key, $ttl, $value)
    {
        $this->doConnect();

        return $this->engine->restore($this->addPrefix($key), $ttl, $value);
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.ShortVariable)
     */
    public function migrate($host, $port, $key, $db, $timeout, $copy = false, $replace = false)
    {
        \trigger_error('Migration is Cache Bundle Engine is not allowed', E_USER_WARNING);

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function time()
    {
        $this->doConnect();

        return $this->engine->time();
    }

    /**
     * {@inheritdoc}
     */
    public function pfAdd($key, array $elements)
    {
        $this->doConnect();

        return $this->engine->pfAdd($this->addPrefix($key), $elements);
    }

    /**
     * {@inheritdoc}
     */
    public function pfCount($key)
    {
        $this->doConnect();

        return $this->engine->pfCount($this->addPrefix($key));
    }

    /**
     * {@inheritdoc}
     */
    public function pfMerge($destkey, array $sourcekeys)
    {
        $this->doConnect();

        return $this->engine->pfMerge($this->addPrefix($destkey), (array) $this->addPrefix($sourcekeys));
    }

    /**
     * {@inheritdoc}
     */
    public function rawCommand($command, ...$arguments)
    {
        $this->doConnect();

        return $this->engine->rawCommand($command, $arguments);
    }

    /**
     * {@inheritdoc}
     */
    public function getMode()
    {
        $this->doConnect();

        return $this->engine->getMode();
    }

    /**
     * Emulating connection on first use instead of on instantiation, similar to memcache.
     * Add server sets arguments doConnect creates the actual connection.
     */
    private function doConnect()
    {
        if ($this->connected || $this->engine instanceof CacheInterface) {
            return;
        }
        \call_user_func_array([$this->engine, 'connect'], $this->serverArgs);
        $this->connected = true;
    }

    // --------------- OUR UNIFIED API BELOW ---------------------

    /**
     * {@inheritdoc}
     */
    public function cAdd($key, $value, $ttl = 0): bool
    {
        return $this->set($key, $value, ['nx', 'ex' => $ttl]);
    }

    /**
     * {@inheritdoc}
     */
    public function cSet($key, $value, $ttl = 0): bool
    {
        return $this->set($key, $value, ['ex' => $ttl]);
    }

    /**
     * {@inheritdoc}
     */
    public function cReplace($key, $value, $ttl = 0): bool
    {
        return $this->set($key, $value, ['xx', 'ex' => $ttl]);
    }

    /**
     * {@inheritdoc}
     */
    public function cGet($key)
    {
        return $this->get($key);
    }

    /**
     * {@inheritdoc}
     */
    public function cDelete($key): bool
    {
        return $this->del($key);
    }

    /**
     * {@inheritdoc}
     */
    public function cIncr($key, $value = 1)
    {
        return $this->incrBy($key, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function cDecr($key, $value = 1)
    {
        return $this->decrBy($key, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function cFlush(): bool
    {
        return $this->flushAll();
    }

    /**
     * {@inheritdoc}
     */
    public function cGetSet($key, $value)
    {
        return $this->getSet($key, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function cListGet($key, $index)
    {
        return $this->lGet($key, $index);
    }

    /**
     * {@inheritdoc}
     */
    public function cListPop($key)
    {
        return $this->lPop($key);
    }

    /**
     * {@inheritdoc}
     */
    public function cListPush($key, $value)
    {
        return $this->rPush($key, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function cListSet($key, $index, $value): bool
    {
        return $this->lSet($key, $index, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function cListCircularGet($key)
    {
        return $this->rpoplpush($key, $key);
    }

    /**
     * {@inheritdoc}
     */
    public function cListClear($key): bool
    {
        return $this->delete($key) > 0;
    }
}
