<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CacheBundle\Session;

use Cyber\CacheBundle\Engine\CacheInterface;
use Cyber\CacheBundle\Engine\EngineFactory;
use InvalidArgumentException;

/**
 * Class CyberCacheSessionHandler.
 */
class CyberCacheSessionHandler implements \SessionHandlerInterface
{
    /**
     * @var CacheInterface
     */
    private $engine;

    private $ttl;

    /**
     * Constructor.
     *
     * List of available options:
     *  * prefix: The prefix to use for the memcache keys in order to avoid collision
     *  * expiretime: The time to live in seconds
     *
     * @param EngineFactory $factory A \Memcache instance
     * @param array<mixed>  $options An associative array of Memcache options
     *
     * @throws \InvalidArgumentException When unsupported options are passed
     */
    public function __construct(EngineFactory $factory, array $options = [])
    {
        $diff = \array_diff(\array_keys($options), ['prefix', 'expiretime']);
        if ($diff) {
            throw new InvalidArgumentException(\sprintf(
                'The following options are not supported "%s"',
                \implode(', ', $diff)
            ));
        }

        $this->ttl    = isset($options['expiretime']) ? (int) $options['expiretime'] : 86400;
        $prefix       = $options['prefix'] ?? 'sf_s';
        $this->engine = $factory->create($prefix);
    }

    /**
     * {@inheritdoc}
     *
     * @param string $savePath
     * @param string $name
     */
    public function open($savePath, $name): bool
    {
        // engine will auto open connection when needed, based on it's own configuration
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function close(): bool
    {
        // we don't want to close cache connection
        return true;
    }

    /**
     * {@inheritdoc}
     *
     * @param string $sesId
     */
    public function read($sesId): string
    {
        return $this->engine->cGet($sesId) ?: '';
    }

    /**
     * {@inheritdoc}
     *
     * @param string $sesId
     * @param string $sesData
     */
    public function write($sesId, $sesData): bool
    {
        return $this->engine->cSet($sesId, $sesData, $this->ttl);
    }

    /**
     * {@inheritdoc}
     *
     * @param string $sesId
     */
    public function destroy($sesId): bool
    {
        if ($this->engine->cGet($sesId)) { // only attempt to delete this when it's there to avoid unnecessary errors
            return $this->engine->cDelete($sesId);
        }

        return true;
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.ShortMethodName)
     *
     * @param int $maxlifetime
     */
    public function gc($maxlifetime): bool
    {
        return true;
    }

    /**
     * @return CacheInterface
     */
    protected function getEngine(): CacheInterface
    {
        return $this->engine;
    }
}
