<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CacheBundle\Engine\Memcache;

use Cyber\CacheBundle\Engine\CacheInterface;
use Cyber\CacheBundle\Engine\EngineTrait;
use InvalidArgumentException;
use Memcache;

/**
 * Class PrefixedMemcache.
 *
 * @SuppressWarnings(PHPMD.CamelCaseParameterName)
 * @SuppressWarnings(PHPMD.ExcessiveClassLength)
 * @SuppressWarnings(PHPMD.TooManyMethods)
 * @SuppressWarnings(PHPMD.TooManyPublicMethods)
 * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
 */
class PrefixedMemcache extends Memcache implements CacheInterface
{
    use EngineTrait;

    /** @var Memcache|mixed|PrefixedMemcache */
    protected $engine;

    /** @var MemcacheList[] */
    protected $lists;

    /**
     * @param Memcache|PrefixedMemcache $engine
     * @param string                    $prefix
     *
     * @throws \RuntimeException when memcache extension is not enabled
     */
    public function __construct($engine, string $prefix)
    {
        if (!$engine instanceof Memcache) {
            throw new InvalidArgumentException('Engine must be instance of \Memcache');
        }
        $this->engine = $engine;
        $this->prefix = $prefix;
    }

    /**
     * {@inheritdoc}
     */
    public function connect($host, $port = null, $timeout = 1): bool
    {
        \trigger_error('Connection is managed by the factory.', E_USER_WARNING);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function pconnect($host, $port = null, $timeout = 1)
    {
        \trigger_error('Connection is managed by the factory.', E_USER_WARNING);

        return true;
    }

    /** @noinspection MoreThanThreeArgumentsInspection */

    /**
     * {@inheritdoc}
     */
    public function addServer(
        $host,
        $port = 11211,
        $persistent = true,
        $weight = null,
        $timeout = 1,
        $retry_interval = 15,
        $status = true,
        callable $failure_callback = null,
        $timeoutms = 0
    ) {
        \trigger_error('Servers are configured through factory.', E_USER_WARNING);

        return true;
    }

    /** @noinspection MoreThanThreeArgumentsInspection */

    /**
     * {@inheritdoc}
     */
    public function setServerParams(
        $host,
        $port = 11211,
        $timeout = 1,
        $retry_interval = 15,
        $status = true,
        callable $failure_callback = null
    ) {
        \trigger_error('Servers are configured through factory.', E_USER_WARNING);

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function setFailureCallback()
    {
        $this->engine->setFailureCallback();
    }

    /**
     * {@inheritdoc}
     */
    public function getServerStatus($host, $port = 11211)
    {
        return $this->engine->getServerStatus($host, $port);
    }

    /**
     * {@inheritdoc}
     */
    public function findServer()
    {
        $this->engine->findServer();
    }

    /**
     * {@inheritdoc}
     */
    public function getVersion()
    {
        return $this->engine->getVersion();
    }

    /** @noinspection MoreThanThreeArgumentsInspection */

    /**
     * {@inheritdoc}
     */
    public function add($key, $var, $flag = 0, $expire = 0)
    {
        return $this->engine->add($this->addPrefix($key), $var, $flag, $expire);
    }

    /** @noinspection MoreThanThreeArgumentsInspection */

    /**
     * {@inheritdoc}
     */
    public function set($key, $var, $flag = 0, $expire = 0)
    {
        return $this->engine->set($this->addPrefix($key), $var, $flag, $expire);
    }

    /** @noinspection MoreThanThreeArgumentsInspection */

    /**
     * {@inheritdoc}
     */
    public function replace($key, $var, $flag = 0, $expire = 0)
    {
        return $this->engine->replace($this->addPrefix($key), $var, $flag, $expire);
    }

    public function cas(): void
    {
        $this->engine->cas();
    }

    public function append(): void
    {
        $this->engine->append();
    }

    /**
     * {@inheritdoc}
     */
    public function prepend()
    {
        return $this->engine->prepend();
    }

    /**
     * {@inheritdoc}
     */
    public function get($key, &$flags = null, &$extra = null)
    {
        // build failing on ubuntu with php7.0 without 3rd parameter, although it is not in specification
        // on windows, failing to call get with 3 params, so just ignore $extra, as it is not supposed to be there to
        // begin with
        return $this->engine->get($this->addPrefix($key), $flags);
    }

    /**
     * {@inheritdoc}
     */
    public function delete($key, $timeout = 0)
    {
        return $this->engine->delete($this->addPrefix($key), $timeout);
    }

    /**
     * {@inheritdoc}
     */
    public function getStats($type = null, $slabid = 0, $limit = 100)
    {
        return $this->engine->getStats($type, $slabid, $limit);
    }

    /**
     * {@inheritdoc}
     */
    public function getExtendedStats($type = null, $slabid = 0, $limit = 100)
    {
        return $this->engine->getExtendedStats($type, $slabid, $limit);
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.CamelCaseParameterName)
     * @SuppressWarnings(PHPMD.CamelCaseVariableName)
     */
    public function setCompressThreshold($thresold, $min_saving = 0.2)
    {
        return $this->engine->setCompressThreshold($thresold, $min_saving);
    }

    /**
     * {@inheritdoc}
     */
    public function increment($key, $value = 1)
    {
        return $this->engine->increment($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function decrement($key, $value = 1)
    {
        return $this->engine->decrement($this->addPrefix($key), $value);
    }

    /**
     * {@inheritdoc}
     */
    public function close()
    {
        \trigger_error('Closing Prefixed Memcached not allowed. Connection managed by factory.', E_USER_WARNING);
    }

    /**
     * {@inheritdoc}
     */
    public function flush()
    {
        return $this->engine->flush();
    }

    // --------------- OUR UNIFIED API BELOW ---------------------
    public function cAdd($key, $value, $ttl = 0): bool
    {
        if ($ttl > 0) {
            return $this->add($key, $value, 0, \time() + $ttl);
        }

        return $this->add($key, $value);
    }

    public function cSet($key, $value, $ttl = 0): bool
    {
        if ($ttl > 0) {
            return $this->set($key, $value, 0, \time() + $ttl);
        }

        return $this->set($key, $value);
    }

    public function cReplace($key, $value, $ttl = 0): bool
    {
        if ($ttl > 0) {
            return $this->replace($key, $value, 0, \time() + $ttl);
        }

        return $this->replace($key, $value);
    }

    public function cGet($key)
    {
        return $this->get($key);
    }

    public function cDelete($key): bool
    {
        return $this->delete($key);
    }

    public function cIncr($key, $value = 1)
    {
        $result = $this->increment($key, $value);

        if (false === $result) { // key does not exist so create it and increment
            $this->set($key, $value);
        }

        return $value;
    }

    public function cDecr($key, $value = 1)
    {
        $result = $this->decrement($key, $value);

        if (false === $result) { // key does not exist so create it and decrement
            $value = -$value;
            $this->set($key, $value);
        }

        return $value;
    }

    public function cFlush(): bool
    {
        return $this->flush();
    }

    /**
     * {@inheritdoc}
     */
    public function cGetSet($key, $value)
    {
        $oldValue = $this->get($key);
        $this->set($key, $value);

        return $oldValue;
    }

    public function cListGet($key, $index)
    {
        $list = $this->getList($key);

        return $list->get($index);
    }

    /**
     * {@inheritdoc}
     */
    public function cListPop($key)
    {
        $list = $this->getList($key);

        return $list->pop();
    }

    /**
     * {@inheritdoc}
     */
    public function cListPush($key, $value)
    {
        $list = $this->getList($key);

        return $list->push($value);
    }

    /**
     * {@inheritdoc}
     */
    public function cListSet($key, $index, $value): bool
    {
        $list = $this->getList($key);

        return $list->set($index, $value);
    }

    /**
     * {@inheritdoc}
     */
    public function cListCircularGet($key)
    {
        $list = $this->getList($key);

        return $list->next();
    }

    /**
     * {@inheritdoc}
     */
    public function cListClear($key): bool
    {
        return $this->getList($key)->clear();
    }

    /**
     * @param string $key
     *
     * @return MemcacheList
     */
    public function getList($key): MemcacheList
    {
        if (!isset($this->lists[$key])) {
            $this->lists[$key] = new MemcacheList($this, $key);
        }

        return $this->lists[$key];
    }

    /**
     * Manually sets list implementation for specific key.
     *
     * Used for testing mostly.
     *
     * @param string       $key
     * @param MemcacheList $list
     */
    public function setList($key, MemcacheList $list)
    {
        $this->lists[$key] = $list;
    }
}
