<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CacheBundle\Session;

use Cyber\CacheBundle\Engine\EngineFactory;
use Cyber\CacheBundle\Engine\Memcache\PrefixedMemcache;
use Cyber\CacheBundle\Session\CyberCacheSessionHandler;
use Memcache;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use ReflectionMethod;
use Tests\Cyber\CacheBundle\Mock\MockMemcache;

/**
 * @group legacy
 */
class CyberCacheSessionHandlerTest extends TestCase
{
    const PREFIX = 'prefix';

    const TTL    = 1000;

    /**
     * @var CyberCacheSessionHandler
     */
    protected $storage;

    /** @var Memcache|MockObject */
    protected $memcache;

    /** @var EngineFactory|MockObject */
    protected $factory;

    public function testOpenSession(): void
    {
        $this->assertTrue($this->storage->open('', ''));
    }

    public function testCloseSession(): void
    {
        $this->assertTrue($this->storage->close());
    }

    public function testReadSession(): void
    {
        $this->memcache
            ->expects($this->once())
            ->method('get')
            ->with(self::PREFIX . '_id');

        $this->assertEquals('', $this->storage->read('id'));
    }

    public function testWriteSession(): void
    {
        $this->memcache
            ->expects($this->once())
            ->method('set')
            ->with(self::PREFIX . '_id', 'data', 0, $this->equalTo(\time() + self::TTL, 2))
            ->will($this->returnValue(true));

        $this->assertTrue($this->storage->write('id', 'data'));
    }

    public function testDestroySession(): void
    {
        $this->memcache
            ->expects($this->once())
            ->method('delete')
            ->with(self::PREFIX . '_id')
            ->will($this->returnValue(true));

        $this->memcache
            ->expects($this->any())
            ->method('get')
            ->with(self::PREFIX . '_id')
            ->willReturnOnConsecutiveCalls(1, false);

        $this->assertTrue($this->storage->destroy('id'));
        $this->assertTrue($this->storage->destroy('id'));
    }

    public function testGcSession(): void
    {
        $this->assertTrue($this->storage->gc(123));
    }

    /**
     * @dataProvider getOptionFixtures
     *
     * @param mixed $options
     * @param mixed $supported
     */
    public function testSupportedOptions($options, $supported): void
    {
        $this->factory = $this->getMockBuilder(EngineFactory::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->factory->expects($this->any())->method('create')
            ->with($options['prefix'] ?? 'sf_s')
            ->willReturn(new PrefixedMemcache($this->memcache, $options['prefix'] ?? 'sf_s'));

        try {
            new CyberCacheSessionHandler($this->factory, $options);
            $this->assertTrue($supported);
        } catch (\InvalidArgumentException $e) {
            $this->assertFalse($supported);
        }
    }

    /**
     * @return array<array<mixed>>
     */
    public function getOptionFixtures(): array
    {
        return [
            [['prefix' => 'session'], true],
            [['expiretime' => 100], true],
            [['prefix' => 'session', 'expiretime' => 200], true],
            [['expiretime' => 100, 'foo' => 'bar'], false],
        ];
    }

    /**
     * @throws \ReflectionException shouldn't
     */
    public function testGetConnection(): void
    {
        $method = new ReflectionMethod($this->storage, 'getEngine');
        $method->setAccessible(true);

        $this->assertInstanceOf('\Memcache', $method->invoke($this->storage));
    }

    protected function setUp(): void
    {
        parent::setUp();
        $this->memcache = $this->getMockBuilder(MockMemcache::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->factory = $this->getMockBuilder(EngineFactory::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->factory->expects($this->any())->method('create')
            ->with(self::PREFIX)
            ->willReturn(new PrefixedMemcache($this->memcache, self::PREFIX));

        $this->storage = new CyberCacheSessionHandler(
            $this->factory,
            ['prefix' => self::PREFIX, 'expiretime' => self::TTL]
        );
    }
}
