<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Manager\CronManager;
use InvalidArgumentException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class CronCommand extends Command
{
    const PERIOD_ARGUMENT_DAILY = 'daily';

    const PERIOD_ARGUMENT_HOURLY = 'hourly';

    private $cronManager;

    /**
     * CronCommand constructor.
     *
     * @param CronManager $cronManager
     */
    public function __construct(CronManager $cronManager)
    {
        parent::__construct();
        $this->cronManager = $cronManager;
    }

    protected function configure(): void
    {
        $this->setName('cyber:cron:run')
            ->addArgument('schedule', InputArgument::OPTIONAL, 'Run tasks only in this schedule')
            ->addOption('all', 'a', InputOption::VALUE_NONE, 'Run all schedules');
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $schedule = $input->getArgument('schedule');
        $all      = $input->getOption('all');
        if (!$schedule && !$all) {
            throw new InvalidArgumentException('Either "schedule" argument or "--all" option must be set');
        }

        $schedules = $schedule ? [$schedule] : $this->cronManager->getSchedules();

        foreach ($schedules as $schedule) {
            $this->executeSchedule($output, $schedule);
        }

        return 0;
    }

    private function executeSchedule(OutputInterface $output, string $schedule): void
    {
        $output->writeln('<info>[cron:' . $schedule . '] <comment>executing...</comment></info>');

        if (!$this->cronManager->hasTasks($schedule)) {
            $output->writeln('    <error>No tasks configured for period "' . $schedule . '"</error>');

            return;
        }
        $this->cronManager->executeScheduleTasks($schedule);

        $output->writeln('<info>[cron:' . $schedule . '] <comment>executed.</comment></info>');
    }
}
