<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CronBundle\Manager;

use Cyber\CacheBundle\Engine\Memcache\PrefixedMemcache;
use Cyber\CronBundle\Component\CronTaskInterface;
use Cyber\CronBundle\Entity\CronTaskInfo;
use Cyber\CronBundle\Manager\CronManager;
use DateTime;
use Doctrine\Common\EventManager;
use Doctrine\DBAL\Connection;
use Doctrine\ORM\Configuration;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Mapping\ClassMetadataFactory;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use ReflectionException;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CronManagerTest extends TestCase
{
    /** @var \PHPUnit_Framework_MockObject_MockObject|PrefixedMemcache */
    private $memcache;

    /** @var EntityManager|\PHPUnit_Framework_MockObject_MockObject */
    private $em;

    /** @var EntityRepository<CronTaskInfo>|\PHPUnit_Framework_MockObject_MockObject */
    private $repo;

    /** @var LoggerInterface|\PHPUnit\Framework\MockObject\MockObject */
    private $logger;

    /** @var CronManager */
    private $cronManager;

    /** @var EventDispatcherInterface|\PHPUnit_Framework_MockObject_MockObject */
    private $eventDispatcher;

    /**
     * {@inheritdoc}
     */
    public function setUp(): void
    {
        $this->memcache = $this->getMockBuilder(PrefixedMemcache::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->em       = $this->getMockBuilder(EntityManager::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->em->expects($this->any())
            ->method('isOpen')
            ->willReturn('testFailureTaskDBReset' === $this->getName() ? false : true);

        $this->repo = $this->getMockBuilder(EntityRepository::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->em->expects($this->any())
            ->method('getRepository')
            ->willReturn($this->repo);

        $this->logger = $this->getMockBuilder(LoggerInterface::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->eventDispatcher = $this->getMockBuilder(EventDispatcherInterface::class)
            ->disableOriginalConstructor()
            ->getMock();

        $metadata = $this->getMockBuilder(ClassMetadata::class)->disableOriginalConstructor()->getMock();

        $this->em->expects($this->any())
            ->method('getClassMetadata')
            ->willReturn($metadata);

        $this->cronManager = new CronManager($this->memcache, $this->em, CronTaskInfo::class, $this->eventDispatcher);
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testUndefinedSchedule(): void
    {
        /** @var CronTaskInterface|\PHPUnit_Framework_MockObject_MockObject $task */
        $task = $this->getMockBuilder(CronTaskInterface::class)
            ->getMock();
        $task->expects($this->once())
            ->method('getSchedule')
            ->willReturn('invalid');

        $this->cronManager->addScheduledTask($task);
        $this->cronManager->hasTasks(CronTaskInterface::class); // trigger resolution of tasks
    }

    public function testAddScheduledTaskClosureAndFindIt(): void
    {
        /** @var CronTaskInterface|\PHPUnit_Framework_MockObject_MockObject $task */
        $task = $this->getMockBuilder(CronTaskInterface::class)
            ->getMock();
        $task->expects($this->once())
            ->method('getSchedule')
            ->willReturn('hourly');

        $this->cronManager->setScheduleIntervals(['hourly' => '1 hour']);
        $this->cronManager->addScheduledTaskClosure(function () use ($task) {
            return $task;
        });
        $found = $this->cronManager->findTask(\get_class($task));
        $this->assertSame($task, $found);
    }

    public function testExecuteScheduleTasksOfNewTask(): void
    {
        /** @var CronTaskInterface|\PHPUnit_Framework_MockObject_MockObject $task */
        $task = $this->getMockBuilder(CronTaskInterface::class)
            ->getMock();

        $task->expects($this->once())
            ->method('getSchedule')
            ->willReturn('hourly');

        $task->expects($this->once())
            ->method('execute');

        $this->memcache->expects($this->once())
            ->method('cAdd')
            ->with(\get_class($task))
            ->willReturn(true);

        $persistedTask = null;
        $this->em->expects($this->once())
            ->method('persist')
            ->with($this->callback(function ($arg) use (&$persistedTask) {
                $persistedTask = $arg;

                return $arg instanceof CronTaskInfo;
            }));

        $task->expects($this->once())
            ->method('execute');

        $this->eventDispatcher
            ->expects($this->exactly(2))
            ->method('dispatch');

        $this->em->expects($this->once())
            ->method('find')
            ->willReturnCallback(function () use (&$persistedTask) {
                return $persistedTask;
            });

        $this->cronManager->setScheduleIntervals(['hourly' => '1 hour']);
        $this->cronManager->addScheduledTask($task);
        $this->cronManager->executeScheduleTasks('hourly');
    }

    public function testExecuteScheduleTasksFailure(): void
    {
        /** @var CronTaskInterface|\PHPUnit_Framework_MockObject_MockObject $task */
        $task = $this->getMockBuilder(CronTaskInterface::class)
            ->getMock();

        $taskInfo = new CronTaskInfo();
        $taskInfo->setLastRun(new DateTime('-2 hours'));

        $this->repo->expects($this->once())
            ->method('findOneBy')
            ->willReturn($taskInfo);

        $this->memcache->expects($this->once())
            ->method('cAdd')
            ->with(\get_class($task))
            ->willReturn(true);

        $this->em->expects($this->never())
            ->method('persist')
            ->with($this->callback(function ($arg) {
                return $arg instanceof CronTaskInfo;
            }));

        $task->expects($this->once())
            ->method('getSchedule')
            ->willReturn('hourly');

        $task->expects($this->once())
            ->method('execute')
            ->willReturn(false);

        $this->eventDispatcher
            ->expects($this->exactly(2))
            ->method('dispatch');

        $this->em->expects($this->once())
            ->method('find')
            ->willReturn($taskInfo);

        $this->cronManager->setScheduleIntervals(['hourly' => '1 hour']);
        $this->cronManager->addScheduledTask($task);
        $this->cronManager->executeScheduleTasks('hourly');

        $this->assertTrue($taskInfo->getIsDisabled());
    }

    public function testDispatchEvents(): void
    {
        /** @var CronTaskInfo|\PHPUnit_Framework_MockObject_MockObject $taskInfo */
        $taskInfo = $this->getMockBuilder(CronTaskInfo::class)->getMock();

        $this->eventDispatcher
            ->expects($this->exactly(3))
            ->method('dispatch');

        $this->invokePrivateMethod($this->cronManager, 'dispatchTaskStartEvent', [$taskInfo]);
        $this->invokePrivateMethod($this->cronManager, 'dispatchTaskFinishedEvent', [$taskInfo]);
        $this->invokePrivateMethod($this->cronManager, 'dispatchTaskFailureEvent', [$taskInfo]);
    }

    public function testFailureTaskDBReset(): void
    {
        $configuration = $this->getMockBuilder(Configuration::class)->getMock();
        $connection    = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();
        $eventManager  = $this->getMockBuilder(EventManager::class)->getMock();
        $factoryClass  = ClassMetadataFactory::class;

        $configuration->method('getMetadataDriverImpl')->willReturn(true);
        $connection->method('getEventManager')->willReturn($eventManager);
        $configuration->method('getClassMetadataFactoryName')->willReturn($factoryClass);
        $configuration->method('getProxyDir')->willReturn('a');
        $configuration->method('getProxyNamespace')->willReturn('a');

        $this->em->expects($this->once())->method('getConnection')
            ->willReturn($connection);
        $this->em->expects($this->once())->method('getConfiguration')
            ->willReturn($configuration);

        $this->invokePrivateMethod($this->cronManager, 'ensureDbObjectManagerOpen', []);
    }

    /**
     * @param object       $object
     * @param string       $methodName
     * @param array<mixed> $parameters
     *
     * @return mixed
     */
    private function invokePrivateMethod(&$object, string $methodName, array $parameters = [])
    {
        try {
            $reflection = new ReflectionClass(\get_class($object));

            $method = $reflection->getMethod($methodName);
            $method->setAccessible(true);
        } catch (ReflectionException $e) {
            $this->fail($e->getMessage());
        }

        return $method->invokeArgs($object, $parameters);
    }
}
