<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Manager;

use Cyber\CacheBundle\Engine\CacheInterface;
use Cyber\CronBundle\Component\CronTaskInterface;
use Cyber\CronBundle\CyberCronEvents;
use Cyber\CronBundle\Entity\CronTaskInfo;
use Cyber\CronBundle\Event\CronTaskInfoEvent;
use DateTime;
use Doctrine\DBAL\Exception\ConnectionException;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ObjectManager;
use Exception;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class CronManager
{
    /** @var null|LoggerInterface */
    private $logger;

    /** @var CacheInterface */
    private $cache;

    /** @var ObjectManager */
    private $objectManager;

    /**
     * @var string
     * @phpstan-var class-string
     */
    private $entityClass;

    /** @var array<string, string> */
    private $scheduleIntervals;

    /** @var CronTaskInterface[][] */
    private $scheduledTasks = [];

    /** @var CronTaskInterface[] */
    private $taskClassMap = [];

    /** @var \Closure[] */
    private $taskClosures = [];

    /** @var EventDispatcherInterface */
    private $eventDispatcher;

    /** @var bool */
    private $resolved = false;

    /**
     * CronManager constructor.
     *
     * @param CacheInterface           $cache
     * @param ObjectManager            $objectManager
     * @param string                   $entityClass
     * @param EventDispatcherInterface $eventDispatcher
     * @param null|LoggerInterface     $logger
     *
     * @phpstan-param class-string     $entityClass
     */
    public function __construct(
        CacheInterface $cache,
        ObjectManager $objectManager,
        string $entityClass,
        EventDispatcherInterface $eventDispatcher,
        LoggerInterface $logger = null
    ) {
        $this->logger          = $logger;
        $this->cache           = $cache;
        $this->objectManager   = $objectManager;
        $this->entityClass     = $entityClass;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param array<string, string> $scheduleIntervals
     */
    public function setScheduleIntervals(array $scheduleIntervals): void
    {
        $this->scheduleIntervals = $scheduleIntervals;
    }

    /**
     * @return string[]
     */
    public function getSchedules(): array
    {
        return \array_keys($this->scheduleIntervals);
    }

    /**
     * @param \Closure $task
     */
    public function addScheduledTaskClosure(\Closure $task): void
    {
        $this->taskClosures[] = $task;
    }

    /**
     * @param CronTaskInterface $task
     *
     * @throws \InvalidArgumentException when $task defines an invalid schedule
     */
    public function addScheduledTask(CronTaskInterface $task): void
    {
        $this->addScheduledTaskClosure(function () use ($task) {
            return $task;
        });
    }

    public function hasTasks(string $schedule): bool
    {
        $this->resolveTasks();

        return isset($this->scheduledTasks[$schedule]);
    }

    public function executeScheduleTasks(string $schedule): bool
    {
        $this->resolveTasks();
        $schedule = $this->normalizeScheduleName($schedule);
        $context  = ['schedule' => $schedule];
        if (!isset($this->scheduledTasks[$schedule])) {
            $this->log('No tasks for schedule: ' . $schedule, $context);

            return false;
        }

        $this->log('Running tasks in schedule: ' . $schedule, $context);

        foreach ($this->scheduledTasks[$schedule] as $service) {
            $this->executeTask($service, $this->scheduleIntervals[$schedule], $context);
        }

        return true;
    }

    /**
     * @param class-string $taskClass
     *
     * @return null|CronTaskInterface
     */
    public function findTask(string $taskClass): ?CronTaskInterface
    {
        $this->resolveTasks();

        return $this->taskClassMap[$taskClass] ?? null;
    }

    /**
     * Executes cron task.
     *
     * @param CronTaskInterface    $taskService
     * @param string               $interval
     * @param array<string, mixed> $context
     *
     * @throws \Exception
     */
    private function executeTask(CronTaskInterface $taskService, string $interval, array $context): void
    {
        $taskServiceId      = \get_class($taskService);
        $context['service'] = $taskServiceId;
        $task               = $this->getTaskIfCanBeExecuted($interval, $context);
        if (!$task) {
            return;
        }

        $this->dispatchTaskStartEvent($task);

        // mark as running
        $task
            ->setLastRun(new DateTime())
            ->setIsRunning(true);

        try {
            $this->objectManager->flush();
        } catch (\Exception $e) {
            //failed to update task date in db we should not proceed.
            $this->log('DB update failed ', $context, 'critical');

            return;
        }

        try {
            if (false === $taskService->execute()) {
                $this->log(
                    '[' . $taskServiceId . '] execution returned FALSE, disabling future executions.',
                    $context,
                    'critical'
                );
                $task->setIsDisabled(true);
            }
        } catch (\Throwable $exception) {
            $context['exception'] = $exception;
            $this->log(
                '[' . $taskServiceId . '] Failed. Uncaught exception. Marked as disabled.',
                $context,
                'critical'
            );
            // don't mark task as disabled if exception occurs for short time
            if (!$this->isTransientException($exception)) {
                $task->setIsDisabled(true);
            }
        } finally {
            $disabled = $this->finishExecution($task, $taskServiceId);
        }

        if (true === $disabled) {
            $this->dispatchTaskFailureEvent($task);

            return;
        }

        $this->dispatchTaskFinishedEvent($task);
    }

    /**
     * Returns task entity if task can be executed.
     *
     * @param string       $interval interval representation acceptable by strtotime() (ex. 60 minutes will cause to
     *                               return true if task hasn't been ran in last 60 minutes)
     * @param array<mixed> $context  context for the task, must include at least key 'service' indicating the service
     *                               name
     *                               (passed as logger context on error)
     *
     * @throws Exception
     *
     * @return null|CronTaskInfo
     */
    private function getTaskIfCanBeExecuted(string $interval, array $context): ?CronTaskInfo
    {
        $taskServiceId = $context['service'];

        // skip if locked for processing by someone
        if (!$this->cache->cAdd($taskServiceId, 'true', 60)) {
            $this->log('[' . $taskServiceId . '] locked for processing by someone. Skipped.', $context);

            return null;
        }

        /** @var null|CronTaskInfo $task */
        $task = $this->objectManager->getRepository($this->entityClass)->findOneBy(['serviceId' => $taskServiceId]);
        if (null === $task) {
            // if it first run, create new record
            /** @var CronTaskInfo $newEntity */
            $newEntity = new $this->entityClass();
            $task      = $newEntity->setServiceId($taskServiceId);
            $this->objectManager->persist($task);

            return $task;
        }

        // silently skip if disabled
        if ($task->getIsDisabled()) {
            return null;
        }
        // skip currently running tasks
        if ($task->getIsRunning()) {
            $this->log('[' . $taskServiceId . '] still running. Skipped.', $context);

            return null;
        }
        $dateLimit = new DateTime();
        $dateLimit
            ->modify('-' . $interval)
            ->modify('+30 seconds'); //add 30 seconds for potential latency, cron can only run once a minute anyway.
        // skip already executed tasks
        if ($task->getLastRun() > $dateLimit) {
            $this->log('[' . $taskServiceId . '] already has been executed in current interval. Skipped.', $context);

            return null;
        }

        return $task;
    }

    /**
     * Check if exception is transient.
     *
     * @param \Throwable $exception
     *
     * @return bool
     */
    private function isTransientException(\Throwable $exception): bool
    {
        return $exception instanceof ConnectionException;
    }

    /**
     * @param string               $message
     * @param array<string, mixed> $context
     * @param string               $level
     */
    private function log(string $message, array $context = [], string $level = 'info'): void
    {
        if ($this->logger) {
            $this->logger->$level($message, $context);
        }
    }

    private function normalizeScheduleName(string $schedule): string
    {
        return \mb_strtolower($schedule);
    }

    /**
     * @param CronTaskInfo $task
     */
    private function dispatchTaskStartEvent(CronTaskInfo $task): void
    {
        $this->eventDispatcher->dispatch(new CronTaskInfoEvent($task), CyberCronEvents::TASK_START);
    }

    /**
     * @param CronTaskInfo $task
     */
    private function dispatchTaskFinishedEvent(CronTaskInfo $task): void
    {
        $this->eventDispatcher->dispatch(new CronTaskInfoEvent($task), CyberCronEvents::TASK_FINISHED);
    }

    /**
     * @param CronTaskInfo $task
     */
    private function dispatchTaskFailureEvent(CronTaskInfo $task): void
    {
        $this->eventDispatcher->dispatch(new CronTaskInfoEvent($task), CyberCronEvents::TASK_FAILURE);
    }

    /**
     * @throws \Doctrine\ORM\ORMException
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    private function ensureDbObjectManagerOpen(): void
    {
        if ($this->objectManager instanceof EntityManagerInterface) {
            if ($this->objectManager->isOpen()) {
                return;
            }

            $this->objectManager = EntityManager::create(
                $this->objectManager->getConnection(),
                $this->objectManager->getConfiguration()
            );
        }

        // other managers not yet implemented
    }

    private function resolveTasks(): void
    {
        if ($this->resolved) {
            return;
        }

        $this->resolved = true;

        foreach ($this->taskClosures as $closure) {
            /** @var CronTaskInterface $task */
            $task     = $closure();
            $schedule = $task->getSchedule();
            $schedule = $this->normalizeScheduleName($schedule);

            if (!isset($this->scheduleIntervals[$schedule])) {
                throw new InvalidArgumentException(\sprintf(
                    'The schedule "%s" is not defined in configuration cyber_cron.schedules',
                    $schedule
                ));
            }
            $this->scheduledTasks[$schedule][]     = $task;
            $this->taskClassMap[\get_class($task)] = $task;
        }
    }

    private function finishExecution(CronTaskInfo &$task, string $taskServiceId): bool
    {
        $disabled = $task->getIsDisabled();

        try {
            $this->ensureDbObjectManagerOpen();
            if (!$this->objectManager->contains($task)) {
                // if not managed re-fetch from db
                $className = \get_class($task);
                $class     = $this->objectManager->getClassMetadata($className);
                // fetch fresh task, if we fail for some reason continue with current task.
                $task = $this->objectManager->find($className, $class->getIdentifierValues($task)) ?: $task;
            }
            // mark as finished
            $task->setIsRunning(false);
            $task->setIsDisabled($disabled);
            $this->objectManager->flush();
        } catch (\Exception $e) {
            $context['exception'] = $e;
            $this->log(
                '[' . $taskServiceId . '] Failed to flush DB.',
                $context,
                'critical'
            );
        }

        return $disabled;
    }
}
