# About
Simple task queue with Symfony and Cron daemon. This bundle requires Memchache to be installed.

# Configuration

This bundle uses the following configuration structure. You should configure all options as needed by your project. You can find this configuration wth the command
```bash
$ php bin/console config:dump-reference cyber_cron
```

```yaml
cyber_cron:
    memcache:
        prefix: cron
    orm: # Required
        manager: doctrine.orm.entity_manager
        task_entity_class: ~ # Required
    schedules:
        # Prototype: expression for DateTime->modify, period how frequently to execute the task
        # Example - hourly: '1 hour' # creates a scheduile for hourly execution
        name:                 ~
```

> 2 schedules are configured by default: `daily` and `hourly` and you do need to add those under `schedules`

## Sample Configuration
First of all, you need to create an entity to store the information about tasks. This entity should extend the CronTaskInfo entity from the bundle.
Ex. */src/AppBundle/Entity/Cron.php*

```php
<?php
namespace AppBundle\Entity;

use Cyber\CronBundle\Entity\CronTaskInfo;
use Doctrine\ORM\Mapping as ORM;

/**
 * Cron
 *
 * @ORM\Table(name="cron_task_info")
 * @ORM\Entity()
 */
class Cron extends CronTaskInfo
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;
}
``` 

After that, you need to make migrations:
```bash
$ php bin\console doctrine:migrations:diff
$ php bin\console doctrine:migrations:migrate
```

Then you need to configure the bundle:
```yml
cyber_cron:
    orm:
        task_entity_class: 'AppBundle\Entity\Cron'
    schedules:
        weekly: '1 week'
```

Now you can create you first task. 

> The tasks support auto-configuration, so if you have default global autoconfigure **ON** the only thing you need to 
> do is implement `CronTaskInterface`. 

```php
<?php
// src/AppBundle/Services/CronTasker.php
namespace AppBundle\Services;

use Cyber\CronBundle\Component\CronTaskInterface;

class MyCronTask implements CronTaskInterface
{
    public function execute(): ?bool
    {
        // Do the work of the task here
        print_r("This is my first task.");
        // return FALSE or throw exception on failure
    }
    
    public function getSchedule(): string {
        // on which schedule this task should run; valid values are 'hourly', 'daily' or any schedule name from
        // configuration above
        return 'weekly';
    }
}
```

If you do not have global autoconfigure enabled, a service configuration for this task (*/app/config/services.yml*). 
```yaml
services:
    AppBundle\Services\MyCronTask:
        autoconfigure: true
```

Now you can test the task execution with command:
```bash
$ php bin/console cyber:cron:run weekly -vv
```

Now you need to add this task to your system's cron configuration:

```bash
0 * * * * php bin/console --env=prod cyber:cron:run --all > /var/log/cyber-cron.log
```
> The above is extremely condensed example. On your production systems you may need to provide full paths to executables
> and configure proper working directory. Also keep in mind cron should execute under user that runs the web server
> behind your application, otherwise cache permission conflicts may occur. 

The cron schedule should be configured to run as frequently as your most frequent task. If your cron runs only once an
hour, all tasks with schedules less than 1 hour would still execute only once in an hour, because they are not being
triggered in a timely fashion. 

# Events
CronBundle generates 3 types of events when the task is executed.
* Before Execution
* After successful completion
* After unsuccessful completion

You need to create a listener file (*listener.php) to subscribe to these events:
```php
class YourListener implements EventSubscriberInterface
{
    public static function getSubscribedEvents(): array
    {
        return [
            CyberCronEvents::TASK_START    => 'taskStart',
            CyberCronEvents::TASK_FINISHED => 'taskFinished',
            CyberCronEvents::TASK_FAILURE  => 'taskFailure',
        ];
    }

    /**
     * @param CronTaskInfoEvent $event
     */
    public function taskStart(CronTaskInfoEvent $event): void
    {
    }

    /**
     * @param CronTaskInfoEvent $event
     */
    public function taskFinished(CronTaskInfoEvent $event): void
    {
    }

    /**
     * @param CronTaskInfoEvent $event
     */
    public function taskFailure(CronTaskInfoEvent $event): void
    {
    }
}
```

And register it in the configuration file (*.yml):

```yaml
services:
    path\to\file\YourListener:
        tags:
            - { name: kernel.event_subscriber }
```

# Multi Server Environment
This bundle leverages memcache for locking purposes. So if 2 servers try to run same task at the same time only one will
succeed. Which means that you can safely run this on any number of servers simultaneously and be sure that tasks would
not get repeated unnecessarily. 

If you do have very large number of servers we recommend adding a `RANDOM_DELAY=15` to your cron file, so that 
cron tasks start with a random delay of up to 15 minutes on each of the servers. This will somewhat mitigate race 
conditions where all servers will attempt to execute the same task on the same second of the day. 
