<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Manager;

use DateTime;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class TaskScheduler
{
    /** @var array<string, string> */
    private $scheduleIntervals;

    private $taskRegistry;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    private $locker;

    public function __construct(TaskRegistry $taskRegistry, TaskLocker $locker, LoggerInterface $logger = null)
    {
        $this->scheduleIntervals = [];
        $this->taskRegistry      = $taskRegistry;
        $this->logger            = $logger ?? new NullLogger();
        $this->locker            = $locker;
    }

    /**
     * @return array<string, string>
     */
    public function getScheduleIntervals(): array
    {
        return $this->scheduleIntervals;
    }

    /**
     * @param array<string, string> $scheduleIntervals
     */
    public function setScheduleIntervals(array $scheduleIntervals): void
    {
        $this->scheduleIntervals = $scheduleIntervals;
    }

    /**
     * Finds and locks next task in the queue.
     *
     * The task must be returned to scheduler via completed() method to schedule next execution.
     *
     * @return \Cyber\CronBundle\Manager\ScheduleContext
     */
    public function nextContext(): ?ScheduleContext
    {
        $repo = $this->taskRegistry->getRepository();

        // must clear before each search otherwise we might get outdated data
        $this->taskRegistry->getObjectManager()->clear($repo->getClassName());

        /** @var \Cyber\CronBundle\Entity\CronTaskInfo[] $taskMetas */
        $taskMetas = $repo->findBy(
            ['isDisabled' => false, 'executingPid' => null],
            ['nextRun' => 'ASC'],
            10
        );
        foreach ($taskMetas as $taskMeta) {
            $id = $taskMeta->getServiceId();
            if (!$taskMeta->shouldRun()) {
                $this->logger->debug(\sprintf('%s does not need to run', $id));
                // since we are in ascending order once we hit one that shouldn't run the rest won't either
                return null;
            }
            $task = $this->taskRegistry->findTask($id);
            if (!$task) {
                $this->logger->warning(\sprintf('Task %s not found. Meta out of sync?', $id));
                continue;
            }

            if ($this->locker->lockTask($taskMeta)) {
                $this->logger->debug(\sprintf('Locked task %s', $id));

                return new ScheduleContext($task, $taskMeta);
            }

            $this->logger->debug(\sprintf('Task %s got locked by another process', $id));
        }

        return null;
    }

    public function completed(ScheduleContext $context, ?string $message): void
    {
        $task     = $context->getTask();
        $meta     = $context->getMeta();
        $messages = $message ? [$message] : [];
        $disable  = $context->failed();
        $nextRun  = null;

        if ($disable) {
            $messages[] = 'Exception: ' . $context->toFailureException()->getMessage();
        }

        $scheduleName = $task->getSchedule();

        $schedule = $this->scheduleIntervals[$scheduleName] ?? null;
        if (!$schedule) {
            $this->logger->error(
                \sprintf('No schedule definition found for name "%s", task will be disabled', $scheduleName),
                ['task' => $meta->getServiceId()]
            );
            // disable the task because we can't really reschedule it
            $disable    = true;
            $messages[] = 'Could not find schedule definition for this task';
        }

        if ($schedule && !$disable) {
            $nextRun = (new DateTime())->modify($schedule);
        }

        $this->locker->unlockTask($meta, $messages, $disable, $nextRun);
    }
}
